<?php
  
  namespace Api\Controllers;
  
  use Api\Core\Controller;
  use Api\Services\UserService;
  use PDO;
  
  class CreditController extends Controller
  {
    public function show()
    {
      $user = UserService::getByParams();
      if (!$user) api()->notFound('USER_NOT_FOUND', 'User not found!');
      
      response()->json([
        'credits' => formatMoney($user['credit']),
        'raw_credits' => $user['credit'],
        'currency' => settings('currency')
      ]);
    }
    
    public function add()
    {
      validate([
				'target_username' => 'required',
        'amount' => 'required|numeric'
      ]);
      
      $user = UserService::getByUsername(input('target_username'));
      if (!$user) api()->notFound('USER_NOT_FOUND', 'User not found!');

      if (input('amount') < 0.01)
        api()->badRequest("INVALID_AMOUNT", "You can't send less than 0.01");
      
      $addUserCredit = db()->prepare("UPDATE Accounts SET credit = credit + ? WHERE id = ?");
      $addUserCredit->execute([input('amount'), $user["id"]]);
      
      $insertCreditHistory = db()->prepare("INSERT INTO CreditHistory (accountID, type, price, creationDate) VALUES (?, ?, ?, ?)");
      $insertCreditHistory->execute(array($user["id"], 1, input('amount'), date("Y-m-d H:i:s")));
      
      $currentCredit = $user['credit'] + input('amount');
      api()->success([
        'credits' => formatMoney($currentCredit),
        'raw_credits' => $currentCredit,
        'currency' => settings('currency')
      ]);
    }
    
    public function remove()
    {
      validate([
				'target_username' => 'required',
        'amount' => 'required|numeric'
      ]);
	    
	    $user = UserService::getByUsername(input('target_username'));
      if (!$user) api()->notFound('USER_NOT_FOUND', 'User not found!');

      if (input('amount') < 0.01)
        api()->badRequest("INVALID_AMOUNT", "You can't send less than 0.01!");

      $updatedCredit = $user['credit'] - input('amount');
      if ($updatedCredit < 0)
        api()->error('NOT_ENOUGH_CREDITS', "This user doesn't have enough credits to remove this amount!");
      
      $removeUserCredit = db()->prepare("UPDATE Accounts SET credit = credit - ? WHERE id = ?");
      $removeUserCredit->execute([input('amount'), $user["id"]]);
      
      $insertCreditHistory = db()->prepare("INSERT INTO CreditHistory (accountID, type, price, creationDate) VALUES (?, ?, ?, ?)");
      $insertCreditHistory->execute(array($user["id"], 2, input('amount'), date("Y-m-d H:i:s")));
      
      api()->success([
        'credits' => formatMoney($updatedCredit),
        'raw_credits' => $updatedCredit,
        'currency' => settings('currency')
      ]);
    }
	  
	  public function send()
	  {
		  // Validate input fields
		  validate([
			  'amount' => 'required|numeric',
			  'sender_username' => 'required',
			  'target_username' => 'required',
		  ]);
			
			if (moduleSettings('credit', 'canSendCredits') == 0) {
				api()->error('CREDIT_DISABLED', 'Credit system is disabled!');
			}
		  
		  // Fetch sender's user data
		  $sender = UserService::getByUsername(input('sender_username'));
		  if (!$sender) api()->notFound('USER_NOT_FOUND', 'User not found!');
		  
		  // Fetch target's user data
		  $target = UserService::getByUsername(input('target_username'));
		  if (!$target) api()->notFound('TARGET_USER_NOT_FOUND', 'Target user not found!');
		  
		  // Prevent the sender from sending credits to themselves
		  if ($sender['id'] == $target['id']) {
			  api()->badRequest("INVALID_TARGET", "You can't send credits to yourself!");
		  }
		  
		  // Validate minimum transfer amount
		  if (input('amount') < 0.01) {
			  api()->badRequest("INVALID_AMOUNT", "You can't send less than 0.01!");
		  }
		  
		  db()->beginTransaction();
		  
		  // Lock sender's account to prevent race conditions
		  $getSenderAccount = db()->prepare("SELECT credit FROM Accounts WHERE id = :sender FOR UPDATE");
		  $getSenderAccount->execute([":sender" => $sender["id"]]);
		  $senderAccount = $getSenderAccount->fetch(PDO::FETCH_ASSOC);
		  
		  // Ensure the sender still has enough balance
		  if (!$senderAccount || $senderAccount["credit"] < input("amount")) {
			  db()->rollBack();
			  api()->error('NOT_ENOUGH_CREDITS', "You don't have enough credits to send this amount!");
		  }
		  
		  // Lock target's account to prevent race conditions
		  $getReceiverAccount = db()->prepare("SELECT credit FROM Accounts WHERE id = :receiver FOR UPDATE");
		  $getReceiverAccount->execute([":receiver" => $target["id"]]);
		  $receiverAccount = $getReceiverAccount->fetch(PDO::FETCH_ASSOC);
		  
		  if (!$receiverAccount) {
			  db()->rollBack();
			  api()->error('TARGET_USER_NOT_FOUND', 'Target user not found!');
		  }
		  
		  // Deduct credit from sender
		  $updateSenderAccount = db()->prepare("UPDATE Accounts SET credit = credit - :amount WHERE id = :sender");
		  $updateSenderAccount->execute([
			  ":amount" => input("amount"),
			  ":sender" => $sender["id"]
		  ]);
		  
		  // Add credit to receiver
		  $updateReceiverAccount = db()->prepare("UPDATE Accounts SET credit = credit + :amount WHERE id = :receiver");
		  $updateReceiverAccount->execute([
			  ":amount"   => input("amount"),
			  ":receiver" => $target["id"]
		  ]);
		  
		  // Insert transaction history for both sender and receiver
		  $insertCreditHistory = db()->prepare("INSERT INTO CreditHistory (accountID, otherAccountID, type, price, creationDate) VALUES (?, ?, ?, ?, ?)");
		  $insertCreditHistory->execute([$sender["id"], $target["id"], 3, input("amount"), date("Y-m-d H:i:s")]);
		  $insertCreditHistory->execute([$target["id"], $sender["id"], 4, input("amount"), date("Y-m-d H:i:s")]);
		  
		  db()->commit();
		  
		  // Calculate and return the updated credit values
		  $currentCredit = $senderAccount['credit'] - input('amount');
		  $currentTargetCredit = $receiverAccount['credit'] + input('amount');
		  
		  api()->success([
			  'current_credits' => formatMoney($currentCredit),
			  'current_raw_credits' => $currentCredit,
			  'target_current_credits' => formatMoney($currentTargetCredit),
			  'target_current_raw_credits' => $currentTargetCredit,
			  'currency' => settings('currency')
		  ]);
	  }
	  
	  public function set()
    {
      validate([
				'target_username' => 'required',
        'amount' => 'required|numeric'
      ]);
	    
	    $user = UserService::getByUsername(input('target_username'));
      if (!$user) api()->notFound('USER_NOT_FOUND', 'User not found!');

      if (input('amount') < 0)
        api()->badRequest("INVALID_AMOUNT", "You can't set less than 0!");
      
      $setUserCredit = db()->prepare("UPDATE Accounts SET credit = ? WHERE id = ?");
      $setUserCredit->execute([input('amount'), $user["id"]]);
      
      $currentCredit = input('amount');
      api()->success([
        'credits' => formatMoney($currentCredit),
        'raw_credits' => $currentCredit,
        'currency' => settings('currency')
      ]);
    }
  }