<?php
  
  namespace Api\Controllers\Store;
  
  use Api\Core\Controller;
  use Api\Core\Jobs\JobDispatcher;
  use Api\Jobs\SendDiscordWebhook;
  use Api\Services\DeliveryService;
  use Api\Services\UserService;
  use PDO;
  
  class BuyController extends Controller
  {
    public function buy()
    {
      validate([
        'userID' => 'required',
        'products' => 'required|array',
      ]);
      
      $user = UserService::getByID(input('userID'));
      if (!$user) api()->notFound('USER_NOT_FOUND', 'User not found!');

      $checkoutItems = input('products');
      $products = [];
      $total = 0;
      $subtotal = 0;
      $discount = 0;
			$tax = 0;
      $credits = $user['credit'];

      // Fetch products
      foreach ($checkoutItems as $checkoutItem) {
        $getProduct = db()->prepare("SELECT P.*, C.name as categoryName, C.isCumulative FROM Products P INNER JOIN ProductCategories C ON P.categoryID = C.id WHERE P.id = ?");
        $getProduct->execute(array($checkoutItem["id"]));
        if ($getProduct->rowCount() > 0) {
          $product = $getProduct->fetch(PDO::FETCH_ASSOC);
	        
	        // Prevent multiple products for cumulative categories
	        if ($product["isCumulative"] == 1 && in_array($product["categoryID"], array_column($products, 'categoryID'))) {
		        continue;
	        }
         
					// Set Quantity
					if ($product["isCumulative"] == 1) {
						$product["quantity"] = 1;
					} else {
						$product["quantity"] = $checkoutItem["quantity"] ?? 1;
					}
					
					// Check subs
	        $product["subscription"] = false;
	        if ($product["isCumulative"] == 1) {
		        $subs = db()->prepare("SELECT * FROM Subscriptions S INNER JOIN Products P ON P.id = S.productID WHERE P.categoryID = ? AND accountID = ? AND (endsAt > ? OR endsAt = '1000-01-01 00:00:00') ORDER BY P.priority ASC LIMIT 1");
		        $subs->execute(array($product["categoryID"], $user["id"], date('Y-m-d H:i:s')));
		        $product["subscription"] = $subs->fetch(PDO::FETCH_ASSOC);
	        }
	        
	        $variables = db()->prepare("SELECT V.* FROM ProductVariables PV INNER JOIN StoreVariables V ON PV.variableID = V.id WHERE PV.productID = ?");
	        $variables->execute(array($product["id"]));
	        $product["required_variables"] = array_map(function ($variable) {
		        $variable["options"] = [];
		        if ($variable["type"] == "dropdown") {
			        $dropdownOptions = db()->prepare("SELECT * FROM StoreVariableDropdownOptions WHERE variableID = ?");
			        $dropdownOptions->execute(array($variable["id"]));
			        $variable["options"] = $dropdownOptions->fetchAll(PDO::FETCH_ASSOC);
		        }
		        
		        return $variable;
	        }, $variables->fetchAll(PDO::FETCH_ASSOC));
					
					$product["variables"] = $checkoutItem["variables"] ?? [];

          $products[] = $product;
        }
      }

      // Check if the product exists
      if (count($products) == 0) {
        api()->notFound('PRODUCT_NOT_FOUND', 'Product not found!');
      }

      // Check quantity
      foreach ($products as $product) {
        if ($product["quantity"] < 1) {
          api()->error('INVALID_QUANTITY', 'Invalid quantity!');
        }
      }
			
	    // Block downgrading
	    foreach ($products as $product) {
		    if ($product["isCumulative"] == '1' && $product["subscription"] && $product["priority"] >= $product["subscription"]["priority"]) {
			    api()->error('DOWNGRADE_NOT_ALLOWED', 'You cannot downgrade your subscription!');
		    }
	    }
	    
	    // Validate variables for products
	    foreach ($products as $product) {
		    foreach ($product["required_variables"] as $variable) {
			    $providedValue = $product["variables"][$variable["id"]] ?? null;
			    $validVariable = false;
			    
			    if ($providedValue !== null) {
				    switch ($variable["type"]) {
					    case "text":
						    $validVariable = strlen($providedValue) > 0;
						    break;
					    case "url":
						    $validVariable = filter_var($providedValue, FILTER_VALIDATE_URL);
						    break;
					    case "email":
						    $validVariable = filter_var($providedValue, FILTER_VALIDATE_EMAIL);
						    break;
					    case "number":
						    $validVariable = is_numeric($providedValue);
						    break;
					    case "dropdown":
						    $validVariable = in_array($providedValue, array_column($variable["options"], 'value'));
						    break;
				    }
			    }
			    
			    if (!$validVariable) {
				    api()->badRequest('INVALID_VARIABLE', "Invalid variable provided for product ID: " . $product["id"]);
			    }
		    }
	    }
			
			// Check Required Account Link
	    foreach ($products as $product) {
				$requiredLinkedAccounts= db()->prepare("SELECT requiredProviderID FROM ProductRequiredLinkedAccounts WHERE productID = ?");
		    $requiredLinkedAccounts->execute([$product['id']]);
		    $requiredLinkedAccounts = $requiredLinkedAccounts->fetchAll(PDO::FETCH_COLUMN);
				foreach ($requiredLinkedAccounts as $requiredLinkedAccount) {
					$accountLinkingProvider = db()->prepare("SELECT name FROM AccountLinkingProviders WHERE id = ? AND isEnabled = ?");
					$accountLinkingProvider->execute([$requiredLinkedAccount['requiredProviderID'], 1]);
					$accountLinkingProvider = $accountLinkingProvider->fetch();
					if (!$accountLinkingProvider) continue;
					
					$checkLinkedAccounts = db()->prepare("SELECT * FROM LinkedAccounts WHERE accountID = ? AND providerID = ?");
					$checkLinkedAccounts->execute([$user['id'], $requiredLinkedAccount['requiredProviderID']]);
					if ($checkLinkedAccounts->rowCount() == 0) {
						api()->error('REQUIRED_LINKED_ACCOUNT', 'You need to link your ' . $accountLinkingProvider['name'] . ' account!');
					}
				}
			}
	  
			
			// Check required products / category required products
      foreach ($products as $product) {
	      $purchasedProductsIDs = db()->prepare("SELECT productID FROM PurchasedProducts WHERE accountID = ? AND (expiresAt > ? OR expiresAt IS NULL)");
	      $purchasedProductsIDs->execute(array($user["id"], date('Y-m-d H:i:s')));
	      $purchasedProductsIDs = $purchasedProductsIDs->fetchAll(PDO::FETCH_COLUMN);
	      $requiredProductIDs = db()->prepare("SELECT requiredProductID FROM ProductRequiredProducts WHERE productID = ?");
	      $requiredProductIDs->execute(array($product["id"]));
	      $requiredProductIDs = $requiredProductIDs->fetchAll(PDO::FETCH_COLUMN);
	      
	      // Check required products
	      if ($requiredProductIDs) {
		      if (!$purchasedProductsIDs) {
			      api()->error('REQUIRED_PRODUCT', 'There are required products you do not own!');
		      }
		      if ($product["requireOnlyOneProduct"] == '1') {
			      if (empty(array_intersect($requiredProductIDs, $purchasedProductsIDs))) {
				      api()->error('REQUIRED_PRODUCT', 'There are required products you do not own!');
			      }
		      } else {
			      $intersection = array_intersect($purchasedProductsIDs, $requiredProductIDs);
			      if (count($intersection) != count($requiredProductIDs)) {
				      api()->error('REQUIRED_PRODUCT', 'There are required products you do not own!');
			      }
		      }
	      }
				
				// Check category required products
	      $categoryRequiredProductIDs = db()->prepare("SELECT productID FROM ProductCategoryRequiredProducts WHERE categoryID = ?");
	      $categoryRequiredProductIDs->execute(array($product["categoryID"]));
	      $categoryRequiredProductIDs = $categoryRequiredProductIDs->fetchAll(PDO::FETCH_COLUMN);
				if ($categoryRequiredProductIDs && empty(array_intersect($purchasedProductsIDs, $categoryRequiredProductIDs))) {
		      api()->error('REQUIRED_PRODUCT', 'There are required products you do not own!');
	      }
	    }

      // Price calculation
      $products = array_map(function($product) use (&$subtotal) {
				$subs = $product["subscription"];
        $price = $product["price"];
        $discountProducts = explode(",", moduleSettings('store', 'discountProducts'));
        $discountedPriceStatus = ($product["discountedPrice"] != 0 && ($product["discountExpiryDate"] > date("Y-m-d H:i:s") || $product["discountExpiryDate"] == '1000-01-01 00:00:00'));
        $storeDiscountStatus = (moduleSettings('store', 'discount') != 0 && (in_array($product["id"], $discountProducts) || moduleSettings('store', 'discountProducts') == '0') && (moduleSettings('store', 'discountExpiryDate') > date("Y-m-d H:i:s") || moduleSettings('store', 'discountExpiryDate') == '1000-01-01 00:00:00'));
	      $subsDiscountStatus = $subs && $product["priority"] < $subs["priority"];
        if ($discountedPriceStatus || $storeDiscountStatus) {
          $price = ($storeDiscountStatus ? (($product["price"]*(100- moduleSettings('store', 'discount')))/100) : $product["discountedPrice"]);
          $product["discountedPrice"] = $price;
        }
	      
	      if ($subsDiscountStatus) {
		      if ($subs["discountedPrice"] > 0 && ($subs["discountExpiryDate"] > date("Y-m-d H:i:s") || $subs["discountExpiryDate"] == '1000-01-01 00:00:00')) {
			      $subs["price"] = $subs["discountedPrice"];
		      }
					else if (moduleSettings('store', 'discount') != 0 && (in_array($subs["productID"], $discountProducts) || moduleSettings('store', 'discountProducts') == '0') && (moduleSettings('store', 'discountExpiryDate') > date("Y-m-d H:i:s") || moduleSettings('store', 'discountExpiryDate') == '1000-01-01 00:00:00')) {
						$subs["price"] = ($subs["price"] * (100 - moduleSettings('store', 'discount'))) / 100;
					}
					
		      if ($subs["endsAt"] == '1000-01-01 00:00:00') {
			      $product["discountedPrice"] = ($discountedPriceStatus || $storeDiscountStatus ? $product["discountedPrice"] : $product["price"]) - ($subs["price"]);
		      } else {
			      $product["discountedPrice"] = ($discountedPriceStatus || $storeDiscountStatus ? $product["discountedPrice"] : $product["price"]) - ((($subs["price"]) * (strtotime($subs["endsAt"]) - time())) / (strtotime($subs["endsAt"]) - strtotime($subs["startsAt"])));
		      }
					$price = $product["discountedPrice"];
	      }
	      
	      // Add extra fees coming from variables
	      foreach ($product["required_variables"] as $variable) {
		      if ($variable["type"] == "dropdown") {
						$providedValue = $product["variables"][$variable["id"]];
			      $selectedOption = db()->prepare("SELECT * FROM StoreVariableDropdownOptions WHERE variableID = ? AND `value` = ?");
			      $selectedOption->execute(array($variable["id"], $providedValue));
			      $selectedOption = $selectedOption->fetch();
			      if ($selectedOption) {
		          $price += $selectedOption["price"];
			      }
		      }
	      }
				
        $subtotal += ($price * $product["quantity"]);

        return $product;
      }, $products);
	    
	    if (moduleSettings('store', 'tax') > 0 && moduleSettings('store', 'isCreditRequiredOnPurchase') == 0) {
		    $tax = ($subtotal - $discount) * (moduleSettings('store', 'tax') / 100);
	    }

      $total = $subtotal + $tax - $discount;

      // Check user balance
      if ($credits < $total) {
        api()->error('INSUFFICIENT_BALANCE', 'Insufficient balance!');
      }

      // Stock check
      foreach ($products as $product) {
        if ($product["stock"] != -1 && $product["stock"] < $product["quantity"]) {
          api()->error('OUT_OF_STOCK', 'Out of stock!');
        }
      }
      
      // Update user balance
      $updateBalance = db()->prepare("UPDATE Accounts SET credit =  credit - ? WHERE id = ?");
      $updateBalance->execute([$total, $user['id']]);
      $credits -= $total;
      
      // Insert transaction
      $insertCreditHistory = db()->prepare("INSERT INTO CreditHistory (accountID, type, price, creationDate) VALUES (?, ?, ?, ?)");
      $insertCreditHistory->execute(array($user["id"], 2, $total, date("Y-m-d H:i:s")));
      
      // Cashback
      $cashback = moduleIsDisabled('credit') ? 0 : ($total * (moduleSettings('store', 'cashback') / 100));
      if ($cashback > 0) {
        $updateCredit = db()->prepare("UPDATE Accounts SET credit = credit + ? WHERE id = ?");
        $updateCredit->execute(array($cashback, $user["id"]));
        $insertCreditHistory = db()->prepare("INSERT INTO CreditHistory (accountID, type, price, creationDate) VALUES (?, ?, ?, ?)");
        $insertCreditHistory->execute(array($user["id"], 1, $cashback, date("Y-m-d H:i:s")));
        $credits += $cashback;
      }
      
      // Create order
      $createOrder = db()->prepare("INSERT INTO Orders (accountID, coupon, total, discount, subtotal, tax, paymentID, paymentAPI, status, type, credit, earnings, cashback, creationDate) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
      $createOrder->execute(array(
        $user["id"],
        null, // coupon
        $total,
        $discount,
        $subtotal,
				$tax,
        "",
        'credit',
        1,
        1,
        $total,
        0,
        $cashback,
        date("Y-m-d H:i:s")
      ));
      $orderID = db()->lastInsertId();
      $createOrderProducts = db()->prepare("INSERT INTO OrderProducts (orderID, productID, quantity, unitPrice) VALUES (?, ?, ?, ?)");
      foreach ($products as $product) {
        $createOrderProducts->execute(array($orderID, $product["id"], $product["quantity"], $product["discountedPrice"] > 0 ? $product["discountedPrice"] : $product["price"]));
	      $orderProductID = db()->lastInsertId();
	      
	      $insertOrderProductVariableValues = db()->prepare("INSERT INTO OrderProductVariableValues (orderProductID, variableID, value) VALUES (?, ?, ?)");
	      foreach ($product["required_variables"] as $variable) {
		      $insertOrderProductVariableValues->execute(array($orderProductID, $variable["id"], $product["variables"][$variable["id"]]));
	      }
      }

      // Update stock
			$products = array_map(function($product) {
				if ($product["stock"] != -1) {
					$updateStock = db()->prepare("UPDATE Products SET stock = stock - ? WHERE id = ?");
					$updateStock->execute(array($product["quantity"], $product["id"]));
					
					// Update from products array
					$product["stock"] -= (int)$product["quantity"];
				}
				
				return $product;
			}, $products);
			
			$productsForDelivery = [];
			foreach ($products as $product) {
				$variablesForDelivery = [];
				foreach ($product["required_variables"] as $variable) {
					$variablesForDelivery[$variable["identifier"]] = $product["variables"][$variable["id"]];
				}
				$productsForDelivery[] = [
					"id" => $product["id"],
					"quantity" => $product["quantity"],
					"variables" => $variablesForDelivery
				];
			}
			
			DeliveryService::deliver($productsForDelivery, $user["id"], "order", $orderID);
	    
	    foreach ($products as $product) {
				JobDispatcher::dispatch((new SendDiscordWebhook('store.purchase.product', [
					"username" => $user["realname"],
					"product" => $product["name"],
					"category" => $product["categoryName"],
					"quantity" => $product["quantity"],
					"price" => $product["discountedPrice"] > 0 ? $product["discountedPrice"] : $product["price"],
					"currency" => settings('currency'),
				])));
	    }

      /*if (moduleSettings('store', 'sendEmailOnPurchase') != '0') {
        $emailItems = [];
        foreach ($products as $product) {
          $emailItems[] = [
            "name" => $product["name"],
            "quantity" => $product["quantity"],
            "total" => $product["discountedPrice"] > 0 ? $product["discountedPrice"] : $product["price"]
          ];
        }
      }*/

      api()->success([
        'orderID' => $orderID,
        'credits' => $credits,
	      'products' => $products,
	      'cashback' => $cashback,
	      'total' => $total,
	      'subtotal' => $subtotal,
	      'discount' => $discount
      ]);
    }
  }