<?php

namespace Dashboard\Controllers;

use Dashboard\Core\Controller;
use Dashboard\Services\AvatarService;
use Dashboard\Services\CategoryService;
use Dashboard\Services\DiscordBotService;
use Dashboard\Services\ImageCropService;
use Dashboard\Services\RoleService;
use donatj\UserAgent\UserAgentParser;
use Dashboard\Libs\Discord;
use Exception;
use PDO;
use Verot\Upload\Upload;

class AccountController extends Controller
{
    public function __construct()
    {
        abort_perm('MANAGE_ACCOUNTS');
    }

    public function index()
    {

        $search = null;
        if (isset($_GET["search"])) {
            $search = get("search");
        }

        if (isset($_GET["page"])) {
            if (!is_numeric($_GET["page"])) {
                $_GET["page"] = 1;
            }
            $page = intval(get("page"));
        } else {
            $page = 1;
        }

        $visiblePageCount = 5;
        $limit = 50;

        $itemsCount = settings("totalAccountCount");
        $pageCount = ceil($itemsCount / $limit);
        if ($page > $pageCount) {
            $page = 1;
        }
        $visibleItemsCount = $page * $limit - $limit;
        $accounts = db()->query("SELECT * FROM Accounts ORDER BY id DESC LIMIT $visibleItemsCount, $limit");

        if ($search) {
          $accounts = db()->prepare("SELECT * FROM Accounts WHERE id = :searchEqual OR realname LIKE :search OR username LIKE :search OR email LIKE :search ORDER BY id DESC");
          $accounts->execute(array(
            "search" => '%' . $search . '%',
            "searchEqual" => $search
          ));
        }
        $accounts = $accounts->fetchAll(PDO::FETCH_ASSOC);

        return view('accounts.index', compact('accounts', 'page', 'visiblePageCount', 'pageCount'));
    }

    public function staff()
    {
      $viewDashboardPerm = db()->prepare("SELECT id FROM Permissions WHERE name = ?");
      $viewDashboardPerm->execute(array('VIEW_DASHBOARD'));
      $readViewDashboardPerm = $viewDashboardPerm->fetch();

      $superAdminPerm = db()->prepare("SELECT id FROM Permissions WHERE name = ?");
      $superAdminPerm->execute(array('SUPER_ADMIN'));
      $readSuperAdminPerm = $superAdminPerm->fetch();

	    $accounts = db()->prepare("SELECT DISTINCT A.* FROM Accounts A LEFT JOIN AccountRoles AR ON AR.accountID = A.id LEFT JOIN Roles R ON AR.roleID = R.id LEFT JOIN RolePermissions RP ON RP.roleID = R.id LEFT JOIN AccountPermissions AP ON AP.accountID = A.id WHERE (AP.permissionID IN (:superperm, :viewperm) OR (RP.permissionID IN (:superperm, :viewperm) AND (AR.expiryDate = '1000-01-01 00:00:00' OR AR.expiryDate >= NOW()))) ORDER BY A.id ASC");
      $accounts->execute(array(
        "viewperm" => $readViewDashboardPerm["id"],
        "superperm" => $readSuperAdminPerm["id"]
      ));
      $accounts = array_map(function ($account) {
        $account["roles"] = styledRoles(getRoles($account["id"]));

        return $account;
      }, $accounts->fetchAll(PDO::FETCH_ASSOC));

      return view('accounts.staff', compact('accounts'));
    }

    public function create()
    {
				if (settings('loginProvider') != 'default') return view('404');
				
        $roles = db()->query("SELECT * FROM Roles ORDER BY priority ASC");
        $roles = array_map(function ($role) {
          $rolePermissions = db()->prepare("SELECT permissionID FROM RolePermissions WHERE roleID = ?");
          $rolePermissions->execute(array($role["id"]));
          $role["permissions"] = $rolePermissions->fetchAll(PDO::FETCH_COLUMN);

          return $role;
        }, $roles->fetchAll());

        $permissions = db()->query("SELECT * FROM Permissions");
        $permissions = $permissions->fetchAll();

        return view('accounts.create', compact('roles', 'permissions'));
    }

    public function store()
    {
	      if (settings('loginProvider') != 'default') return view('404');
				
        validate([
            'username' => 'required',
            'email' => 'required',
            'password' => 'required',
            'passwordRe' => 'required|same:password',
        ]);

        $usernameValid = db()->prepare("SELECT COUNT(id) FROM Accounts WHERE username = ? LIMIT 1");
        $usernameValid->execute(array(input("username")));
        $usernameCount = $usernameValid->fetchColumn();

        $emailValid = db()->prepare("SELECT COUNT(id) FROM Accounts WHERE email = ? LIMIT 1");
        $emailValid->execute(array(input("email")));
        $emailCount = $emailValid->fetchColumn();

        if (input("credit") == null) {
          set_input("credit", 0);
        }

        if (!validate_username(input("username"))) {
            return back()->flash('error', t__("Please enter a valid username!"));
        } else if ($usernameCount > 0) {
            return back()->flash('error', t__("This username is already in use!"));
        } else if (checkEmail(input("email"))) {
            return back()->flash('error', t__("Please enter a valid email!"));
        } else if ($emailCount > 0) {
            return back()->flash('error', t__("This email is already in use!"));
        } else if (strlen(input("password")) < 4) {
            return back()->flash('error', t__("Password shouldn't be less than 4 characters!"));
        } else if (input("password") != input("passwordRe")) {
            return back()->flash('error', t__("Passwords doesn't match!"));
        } else if (checkBadPassword(input("password"))) {
            return back()->flash('error', t__("Please enter a valid password!"));
        } else {
            // Security Check
            if (!auth()->user()->isSuperAdmin() && auth()->user()->can("MANAGE_ROLES")) {
              if (isset($_POST["roles"])) {
                foreach ($_POST["roles"] as $role) {
                  if ($role == 1) continue;

                  $roleQuery = db()->prepare("SELECT priority FROM Roles WHERE id = ?");
                  $roleQuery->execute(array($role));
                  $roleQuery = $roleQuery->fetch();
                  $isHigher = RoleService::checkHigherRoleByUserId(auth()->user()->id(), $roleQuery["priority"]);
                  if (!$isHigher) {
                    return back()->flash("error", t__('You cannot select a role with a higher priority than your highest role!'));
                  }
                }
              }
              if (isset($_POST["permissions"])) {
                $isDifferentPermissions = RoleService::isDifferentPermissions(auth()->user()->permissions(), $_POST["permissions"]);
                if ($isDifferentPermissions) {
                  return back()->flash("error", t__('You cannot select a permission you do not have!'));
                }
              }
            }

            $password = createPassword(settings("passwordType"), input("password"));

            $insertAccounts = db()->prepare("INSERT INTO Accounts (username, realname, email, password, credit, creationIP, creationDate) VALUES (?, ?, ?, ?, ?, ?, ?)");
            $insertAccounts->execute(array(strtolower(input("username")), input("username"), input("email"), $password, input("credit"), getIP(), datetime()));
            $accountID = db()->lastInsertId();

            if (auth()->user()->can("MANAGE_ROLES")) {
                if (isset($_POST["roles"])) {
                    foreach ($_POST["roles"] as $role) {
                        if ($role == 1)
                            continue;

                        $roleDateSettings = json_decode(input('roleDateSettings'), true);
                        $roleExpiryDate = $roleDateSettings[$role] ?? '1000-01-01 00:00:00';
                        $addRoleToUser = db()->prepare("INSERT INTO AccountRoles (accountID, roleID, expiryDate) VALUES (?, ?, ?)");
                        $addRoleToUser->execute(array($accountID, $role, $roleExpiryDate));
                    }
                }

                if (isset($_POST["permissions"])) {
                    foreach ($_POST["permissions"] as $permission) {
                        $permission = strip_tags($permission);
                        $addPermToUser = db()->prepare("INSERT INTO AccountPermissions (accountID, permissionID) VALUES (?, ?)");
                        $addPermToUser->execute(array($accountID, $permission));
                    }
                }
            }

            createLog("USER_CREATED", [
              'username' => input('username')
            ]);

            return back()->flash("success", t__('Account added successfully.'));
        }
    }

    public function edit($id)
    {
        $account = db()->prepare("SELECT * FROM Accounts WHERE id = ?");
        $account->execute(array($id));
        $account = $account->fetch();

        if (!$account) return view('404');
	    
		    $account["avatar"] = AvatarService::get($account["id"], $account["realname"]);

        $account["discordUserID"] = null;
        if (!moduleIsDisabled('discord')) {
            $discordUser = db()->prepare("SELECT * FROM AccountDiscordData WHERE accountID = ?");
            $discordUser->execute(array($account["id"]));
            $discordUser = $discordUser->fetch();
            if ($discordUser) {
                $account["discordUserID"] = $discordUser["discordUserID"];
            }
        }

        $roles = db()->query("SELECT * FROM Roles ORDER BY priority ASC");
        $roles = array_map(function ($role) {
            $rolePermissions = db()->prepare("SELECT permissionID FROM RolePermissions WHERE roleID = ?");
            $rolePermissions->execute(array($role["id"]));
            $role["permissions"] = $rolePermissions->fetchAll(PDO::FETCH_COLUMN);

            return $role;
        }, $roles->fetchAll());

        $permissions = db()->query("SELECT * FROM Permissions");
        $permissions = $permissions->fetchAll();

        $account["permissions"] = getPermissions($account["id"]);
        if (!auth()->user()->isSuperAdmin() && checkPerm($account, 'SUPER_ADMIN')) {
          abort_403();
        }

        $accountRoles = getRoles($account["id"], true);
        $accountRoleIDs = array_column($accountRoles, 'id');
        if (!auth()->user()->isSuperAdmin() && count($accountRoles) > 0) {
          $isHigher = RoleService::checkHigherRoleByUserId(auth()->user()->id(), $accountRoles[0]["priority"]);
          if (!$isHigher) {
            abort_403(
              null,
              t__('You cannot edit an account with a higher priority than your highest role!')
            );
          }
        }

        $accountPermissions = db()->prepare("SELECT permissionID FROM AccountPermissions WHERE accountID = ?");
        $accountPermissions->execute(array($account["id"]));
        $accountPermissionIDs = $accountPermissions->fetchAll(PDO::FETCH_COLUMN);
	    
		    $linkedAccounts = [];
		    if (DiscordBotService::isBotReady() || !moduleIsDisabled('account_linking')) {
			    $linkedAccounts = db()->prepare("SELECT P.id, P.name, P.slug, L.identifier, L.displayName FROM LinkedAccounts L INNER JOIN AccountLinkingProviders P ON P.id = L.providerID WHERE L.accountID = ?");
			    $linkedAccounts->execute([$account["id"]]);
			    $linkedAccounts = $linkedAccounts->fetchAll(PDO::FETCH_ASSOC);
		    }
		    $account["linkedAccounts"] = $linkedAccounts;

        return view('accounts.edit', compact('account', 'roles', 'permissions', 'accountRoles', 'accountRoleIDs', 'accountPermissionIDs'));
    }

    public function update($id)
    {
        $account = db()->prepare("SELECT * FROM Accounts WHERE id = ?");
        $account->execute(array($id));
        $account = $account->fetch();

        if (!$account) return view('404');
				
				$username = $account["username"];
				$displayName = $account["realname"];
				$email = null;
				$password = null;

        if (settings('loginProvider') == 'default') {
	        validate([
		        'username' => 'required',
	        ]);
					
	        $username = strtolower(input('username'));
	        $displayName = input('username');
	        
	        $usernameValid = db()->prepare("SELECT COUNT(id) FROM Accounts WHERE id != ? AND username = ?");
	        $usernameValid->execute(array($account["id"], $username));
	        $usernameCount = $usernameValid->fetchColumn();
					
	        if ($usernameCount > 0) {
		        return back()->flash('error', t__("This username is already in use!"));
	        }
	        
	        if (!validate_username($displayName)) {
		        return back()->flash('error', t__("Please enter a valid username"));
	        }
        }

        if (input("email")) {
	        $email = input("email");
	        $emailValid = db()->prepare("SELECT COUNT(id) FROM Accounts WHERE id != ? AND email = ?");
	        $emailValid->execute(array($account["id"], $email));
	        $emailCount = $emailValid->fetchColumn();
	        
	        if ($emailCount > 0) {
		        return back()->flash('error', t__("This email is already in use!"));
	        }
	        
	        if (checkEmail(input("email"))) {
		        return back()->flash('error', t__("Please enter a valid email!"));
	        }
        }

        if (input("credit") == null) {
            set_input("credit", $account["credit"]);
        }

        if (settings('loginProvider') == 'default') {
	        if (strlen(input("password")) < 4 && (input("password") != null && input("passwordRe") != null)) {
		        return back()->flash('error', t__("Password shouldn't be less than 4 characters!"));
	        } else if (input("password") != input("passwordRe") && (input("password") != null && input("passwordRe") != null)) {
		        return back()->flash('error', t__("Passwords doesn't match!"));
	        } else if (checkBadPassword(input("password")) && (input("password") != null && input("passwordRe") != null)) {
		        return back()->flash('error', t__('Please enter a valid password!'));
	        }
	        
	        if (input("password") != null && input("passwordRe") != null) {
		        $password = createPassword(settings("passwordType"), input("password"));
	        } else {
		        $password = $account["password"];
	        }
        }
				
        // Security Check
        $account["permissions"] = getPermissions($account["id"]);
				$currentPermissions = $account["permissions"];
        if (!auth()->user()->isSuperAdmin() && checkPerm($account, 'SUPER_ADMIN')) {
          abort_403();
        }

        $accountRoles = getRoles($account["id"], true);
        if (!auth()->user()->isSuperAdmin() && count($accountRoles) > 0) {
          $isHigher = RoleService::checkHigherRoleByUserId(auth()->user()->id(), $accountRoles[0]["priority"]);
          if (!$isHigher) {
            abort_403(
              null,
              t__('You cannot edit an account with a higher priority than your highest role!')
            );
          }
        }
        
        // Security Check
        if (!auth()->user()->isSuperAdmin() && auth()->user()->can('MANAGE_ROLES')) {
          if (isset($_POST["roles"])) {
            foreach ($_POST["roles"] as $role) {
              if ($role == 1) continue;

              $roleQuery = db()->prepare("SELECT priority FROM Roles WHERE id = ?");
              $roleQuery->execute(array($role));
              $roleQuery = $roleQuery->fetch();
              $isHigher = RoleService::checkHigherRoleByUserId(auth()->user()->id(), $roleQuery["priority"]);
              if (!$isHigher) {
                return back()->flash("error", t__('You cannot select a role with a higher priority than your highest role!'));
              }
            }
          }
          if (isset($_POST["permissions"])) {
            $isDifferent = RoleService::isDifferentPermissions(auth()->user()->permissions(), $_POST["permissions"]);
            if ($isDifferent) {
              return back()->flash("error", t__('You cannot select a permission you do not have!'));
            }
          }
        }
				
				if (moduleIsDisabled('email_verification')) {
					$isVerified = $account["isVerified"] == '1';
				} else {
					$isVerified = input('isVerified') == '1';
				}

        if (($displayName != $account["realname"]) || ($email != $account["email"]) || (input("password") != null && input("passwordRe") != null)) {
            $deleteAccountSessions = db()->prepare("DELETE FROM AccountSessions WHERE accountID = ?");
            $deleteAccountSessions->execute(array($id));
        }
        $updateAccounts = db()->prepare("UPDATE Accounts SET username = ?, realname = ?, email = ?, password = ?, credit = ?, isVerified = ? WHERE id = ?");
        $updateAccounts->execute(array($username, $displayName, $email, $password, input("credit"), $isVerified ? '1' : '0', $id));

        if (auth()->user()->can('MANAGE_ROLES')) {
            $removeRolesFromUser = db()->prepare("DELETE FROM AccountRoles WHERE accountID = ?");
            $removeRolesFromUser->execute(array($account["id"]));
            if (isset($_POST["roles"])) {
                foreach ($_POST["roles"] as $role) {
                    if ($role == 1)
                        continue;

                    $roleDateSettings = json_decode(input('roleDateSettings'), true);
                    $roleExpiryDate = $roleDateSettings[$role] ?? '1000-01-01 00:00:00';
                    $addRoleToUser = db()->prepare("INSERT INTO AccountRoles (accountID, roleID, expiryDate) VALUES (?, ?, ?)");
                    $addRoleToUser->execute(array($account["id"], $role, $roleExpiryDate));
                }
            }

            $removePermsFromUser = db()->prepare("DELETE FROM AccountPermissions WHERE accountID = ?");
            $removePermsFromUser->execute(array($account["id"]));
            if (isset($_POST["permissions"])) {
                foreach ($_POST["permissions"] as $permission) {
                    $permission = strip_tags($permission);
                    $addPermToUser = db()->prepare("INSERT INTO AccountPermissions (accountID, permissionID) VALUES (?, ?)");
                    $addPermToUser->execute(array($account["id"], $permission));
                }
            }
        }

        $account["permissions"] = getPermissions($account["id"]);
				$newPermissions = $account["permissions"];
        $updateOnlineAccountsHistory = db()->prepare("UPDATE OnlineAccountsHistory SET type = ? WHERE accountID = ?");
        $updateOnlineAccountsHistory->execute(array(checkStaff($account), $account["id"]));
				
				// Discord Role Update
	      if (!moduleIsDisabled('discord_bot') && DiscordBotService::isBotReady()) {
					// Check Discord Account Link
		      $discordLinkedAccount = db()->prepare("SELECT L.identifier FROM LinkedAccounts L INNER JOIN AccountLinkingProviders P ON P.id = L.providerID WHERE L.accountID = ? AND P.slug = ?");
		      $discordLinkedAccount->execute([$account["id"], 'discord']);
		      $discordLinkedAccount = $discordLinkedAccount->fetch(PDO::FETCH_ASSOC);
		      
		      $siteRoles = db()->query("SELECT discordRoleID FROM Roles WHERE discordRoleID IS NOT NULL AND discordRoleID != ''");
		      $siteRoles = $siteRoles->fetchAll(PDO::FETCH_ASSOC);
					
					if ($discordLinkedAccount && count($siteRoles) > 0) {
						// Send request to LeaderOS Discord Bot API to remove discord roles
						DiscordBotService::unlinkUser($discordLinkedAccount['identifier'], array_column($siteRoles, 'discordRoleID'));
						
						// Wait for a second to ensure the roles are removed before adding new ones
						sleep(1);
						
						// Send request to LeaderOS Discord Bot API to add discord roles
						$userRoles = db()->prepare("SELECT R.discordRoleID, AR.expiryDate FROM AccountRoles AR INNER JOIN Roles R ON AR.roleID = R.id WHERE AR.accountID = ? AND R.discordRoleID IS NOT NULL AND R.discordRoleID != '' AND (AR.expiryDate = '1000-01-01 00:00:00' OR AR.expiryDate > ?)");
						$userRoles->execute([$account["id"], datetime()]);
						$userRoles = array_map(function ($role) {
							return [
								'id' => $role['discordRoleID'],
								'expiryDate' => $role['expiryDate']
							];
						}, $userRoles->fetchAll(PDO::FETCH_ASSOC));
						
						$defaultRole = db()->prepare('SELECT discordRoleID FROM Roles WHERE slug = ? AND discordRoleID IS NOT NULL AND discordRoleID != "" LIMIT 1');
						$defaultRole->execute(['default']);
						$defaultRole = $defaultRole->fetch(PDO::FETCH_ASSOC);
						if ($defaultRole) {
							$userRoles[] = [
								'id' => $defaultRole['discordRoleID'],
								'expiryDate' => '1000-01-01 00:00:00'
							];
						}
						
						$currentDisplayName = db()->prepare("SELECT realname FROM Accounts WHERE id = ?");
						$currentDisplayName->execute([$account["id"]]);
						$currentDisplayName = $currentDisplayName->fetchColumn();
						
						DiscordBotService::linkUser($discordLinkedAccount["identifier"], $currentDisplayName, $userRoles);
					}
	      }
				
				// Discord Role Update for Legacy Bot
        if (!moduleIsDisabled('discord') && moduleSettings('discord', 'roleSyncingStatus') == 1) {
          $accountDiscordData = db()->prepare("SELECT * FROM AccountDiscordData WHERE accountID = ?");
          $accountDiscordData->execute(array($account["id"]));
          $accountDiscordData = $accountDiscordData->fetch();
          if ($accountDiscordData) {
            $discord = new Discord(moduleSettings('discord', 'clientID'), moduleSettings('discord', 'clientSecret'), moduleSettings('discord', 'botToken'));
            $discord->removeRole(moduleSettings('discord', 'guildID'), $accountDiscordData["discordUserID"], moduleSettings('discord', 'syncedRoleID'));
            $discord->giveRole(moduleSettings('discord', 'guildID'), $accountDiscordData["discordUserID"], moduleSettings('discord', 'syncedRoleID'));
          }
        }
      
        createLog("USER_UPDATED", [
	        'username' => $account["username"],
	        'displayName' => $account["realname"],
	        'email' => $account["email"],
					'credits' => $account["credit"],
					'permissions' => $currentPermissions,
	        'new_username' => input('username'),
	        'new_email' => input('email'),
	        'new_credits' => input('credit'),
	        'new_permissions' => $newPermissions
        ]);

        return back()->flash("success", t__('Changes has been saved successfully!'));
    }

    public function show($id)
    {
        $account = db()->prepare("SELECT * FROM Accounts WHERE id = ?");
        $account->execute(array($id));
        $account = $account->fetch();

        if (!$account) return view('404');
	    
	      $account["avatar"] = AvatarService::get($account["id"], $account["realname"]);

        if (count($account) > 0) {
            $account["roles"] = getRoles($account["id"]);

            $lastSeen = db()->prepare("SELECT * FROM OnlineAccountsHistory WHERE accountID = ?");
            $lastSeen->execute(array($account["id"]));
            $lastSeen = $lastSeen->fetch();

            $accountCustomFieldValues = db()->prepare("SELECT ACF.name, ACFV.value FROM AccountCustomFieldValues ACFV INNER JOIN AccountCustomFields ACF ON ACF.id = ACFV.customFieldID WHERE ACFV.accountID = ? AND ACFV.value != ''");
            $accountCustomFieldValues->execute(array($account["id"]));
            $accountCustomFieldValues = $accountCustomFieldValues->fetchAll(PDO::FETCH_ASSOC);

            $siteBannedAccountStatus = db()->prepare("SELECT * FROM BannedAccounts WHERE accountID = ? AND categoryID = ? AND (expiryDate > ? OR expiryDate = ?) ORDER BY expiryDate DESC LIMIT 1");
            $siteBannedAccountStatus->execute(array($account["id"], 1, datetime(), '1000-01-01 00:00:00'));
            $siteBannedAccountStatus = $siteBannedAccountStatus->fetch();

            $supportBannedAccountStatus = db()->prepare("SELECT * FROM BannedAccounts WHERE accountID = ? AND categoryID = ? AND (expiryDate > ? OR expiryDate = ?) ORDER BY expiryDate DESC LIMIT 1");
            $supportBannedAccountStatus->execute(array($account["id"], 2, datetime(), '1000-01-01 00:00:00'));
            $supportBannedAccountStatus = $supportBannedAccountStatus->fetch();

            $commentBannedAccountStatus = db()->prepare("SELECT * FROM BannedAccounts WHERE accountID = ? AND categoryID = ? AND (expiryDate > ? OR expiryDate = ?) ORDER BY expiryDate DESC LIMIT 1");
            $commentBannedAccountStatus->execute(array($account["id"], 3, datetime(), '1000-01-01 00:00:00'));
            $commentBannedAccountStatus = $commentBannedAccountStatus->fetch();

            $statServers = db()->query("SELECT serverName, serverSlug FROM Leaderboards");
            $statServers->execute();
            $statServers = $statServers->fetchAll(PDO::FETCH_ASSOC);

            $statServer = null;
            if (get('leaderboard') != null) {
              $statServer = db()->prepare("SELECT * FROM Leaderboards WHERE serverSlug = ?");
              $statServer->execute([get('leaderboard')]);
              $statServer = $statServer->fetch(PDO::FETCH_ASSOC);
            }

            $statServerTableItems = [];
            $statServerData = [];
            $statServerUser = [];
            $statServerUserRank = 1;

            if ($statServer) {
              $usernameColumn = $statServer["usernameColumn"];
              $mysqlTable = $statServer["mysqlTable"];
              $sorter = $statServer["sorter"];
              $tableTitles = $statServer["tableTitles"];
              $tableData = $statServer["tableData"];
              $statServerTableItems = explode(",", $tableTitles);
              $statServerData = explode(",", $tableData);

              if ($statServer["mysqlServer"] == '0') {
                $statServerUser = db()->prepare("SELECT $usernameColumn,$tableData FROM $mysqlTable WHERE $usernameColumn = ? ORDER BY $sorter DESC LIMIT 1");
                $statServerUser->execute(array($account["username"]));
              }
              else {
                try {
                  $newDB = new PDO("mysql:host=".$statServer["mysqlServer"]."; port=".$statServer["mysqlPort"]."; dbname=".$statServer["mysqlDatabase"]."; charset=utf8", $statServer["mysqlUsername"], $statServer["mysqlPassword"]);
                }
                catch (PDOException $e) {
                  die("<strong>MySQL connection error:</strong> ".utf8_encode($e->getMessage()));
                }
                $statServerUser = $newDB->prepare("SELECT $usernameColumn,$tableData FROM $mysqlTable WHERE $usernameColumn = ? ORDER BY $sorter DESC LIMIT 1");
                $statServerUser->execute(array($account["username"]));
              }
              $statServerUser = $statServerUser->fetch(PDO::FETCH_ASSOC);
              if ($statServer["mysqlServer"] == '0') {
                $userPosition = db()->prepare("SELECT $usernameColumn FROM $mysqlTable ORDER BY $sorter DESC");
                $userPosition->execute();
              }
              else {
                $userPosition = $newDB->prepare("SELECT $usernameColumn FROM $mysqlTable ORDER BY $sorter DESC");
                $userPosition->execute();
              }
              foreach ($userPosition as $readUserPosition) {
                if ($readUserPosition[$usernameColumn] == $account["username"]) {
                  break;
                }
                $statServerUserRank++;
              }
            }

            $chests = db()->prepare("SELECT C.*, P.name as productName, PC.id as categoryID, PC.name as categoryName FROM Chests C INNER JOIN Products P ON C.productID = P.id INNER JOIN ProductCategories PC ON P.categoryID = PC.id WHERE C.accountID = ? AND C.status = ? ORDER BY C.id DESC");
            $chests->execute(array($account["id"], 0));
		        $chests = array_map(function ($chest) {
			        $chestProductVariableValues = db()->prepare("SELECT V.*, CPVV.value FROM ChestProductVariableValues CPVV INNER JOIN StoreVariables V ON V.id = CPVV.variableID  WHERE chestID = ?");
			        $chestProductVariableValues->execute([$chest["id"]]);
			        $chest["variables"] = $chestProductVariableValues->fetchAll(PDO::FETCH_ASSOC);
			        
			        return $chest;
		        }, $chests->fetchAll(PDO::FETCH_ASSOC));

            global $categoryList;
            $chests = CategoryService::getCategoryList($chests, array_column($chests, "categoryID"));

            $tickets = db()->prepare("SELECT S.*, SC.name as categoryName FROM Supports S INNER JOIN SupportCategories SC ON S.categoryID = SC.id WHERE S.accountID = ? ORDER BY S.updateDate DESC LIMIT 50");
            $tickets->execute(array($account["id"]));
            $tickets = $tickets->fetchAll(PDO::FETCH_ASSOC);

            $creditLogs = db()->prepare("SELECT CH.*, A.username, A.realname as otherAccountUsername FROM CreditHistory CH LEFT JOIN Accounts A ON A.id = CH.otherAccountID WHERE CH.accountID = ? ORDER BY CH.id DESC LIMIT 50");
            $creditLogs->execute(array($account["id"]));
            $creditLogs = $creditLogs->fetchAll(PDO::FETCH_ASSOC);

            $orders = db()->prepare("SELECT * FROM Orders WHERE accountID = ? ORDER BY id DESC LIMIT 50");
            $orders->execute(array($account["id"]));
            $orders = $orders->fetchAll(PDO::FETCH_ASSOC);

            $lotteryLogs = db()->prepare("SELECT LH.*, L.title as lotteryTitle, LA.title, LA.awardType, LA.award FROM LotteryHistory LH INNER JOIN LotteryAwards LA ON LH.lotteryAwardID = LA.id INNER JOIN Lotteries L ON LA.lotteryID = L.id WHERE LH.accountID = ? AND LA.awardType != ? ORDER by LH.id DESC LIMIT 50");
            $lotteryLogs->execute(array($account["id"], 3));
            $lotteryLogs = $lotteryLogs->fetchAll(PDO::FETCH_ASSOC);

            $giftLogs = db()->prepare("SELECT PGH.*, PG.name, PG.giftType, PG.gift FROM ProductGiftsHistory PGH INNER JOIN ProductGifts PG ON PGH.giftID = PG.id WHERE PGH.accountID = ? ORDER by PGH.id DESC LIMIT 50");
            $giftLogs->execute(array($account["id"]));
            $giftLogs = $giftLogs->fetchAll(PDO::FETCH_ASSOC);
            $giftLogs = array_map(function ($item) {
              /* gift type */
              if ($item['giftType'] == 1) {
                $product = db()->prepare("SELECT name FROM Products WHERE id = ?");
                $product->execute(array($item["gift"]));
                $product = $product->fetch();

                $item['productName'] = $product["name"];
              } else $item['productName'] = credits($item['gift']);
              return $item;
            }, (array)$giftLogs);

            $chestLogs = db()->prepare("SELECT CH.*, P.name as productName, PC.id as categoryID, PC.name as categoryName, OA.realname as otherAccountUsername FROM ChestsHistory CH INNER JOIN Chests C ON CH.chestID = C.id INNER JOIN Products P ON C.productID = P.id INNER JOIN ProductCategories PC ON P.categoryID = PC.id LEFT JOIN Accounts OA ON OA.id = CH.otherAccountID WHERE CH.accountID = ? ORDER BY CH.id DESC LIMIT 50");
            $chestLogs->execute(array($account["id"]));
		        $chestLogs = $chestLogs->fetchAll(PDO::FETCH_ASSOC);

            $chestLogs = CategoryService::getCategoryList($chestLogs, array_column($chestLogs, "categoryID"));

            $bazaarLogs = db()->prepare("SELECT BH.*, BI.name as itemName, BI.price as itemPrice FROM BazaarHistory BH INNER JOIN BazaarItems BI ON BH.itemID = BI.id WHERE BH.accountID = ? ORDER BY BH.id DESC LIMIT 50");
            $bazaarLogs->execute(array($account["id"]));
            $bazaarLogs = $bazaarLogs->fetchAll(PDO::FETCH_ASSOC);

            $applications = db()->prepare("SELECT AP.id, AF.title, AP.reason, AP.status FROM Applications AP INNER JOIN Accounts A ON A.id = AP.accountID INNER JOIN ApplicationForms AF ON AF.id = AP.formID WHERE AP.accountID = ? ORDER BY AP.id DESC LIMIT 50");
            $applications->execute(array($account["id"]));
            $applications = $applications->fetchAll(PDO::FETCH_ASSOC);
	        
		        $loginLogs = db()->prepare("SELECT S.* FROM AccountSessions S INNER JOIN Accounts A ON A.id = S.accountID WHERE S.accountID = ? ORDER BY S.id DESC LIMIT 50");
		        $loginLogs->execute(array($account["id"]));
	          $uaParser = new UserAgentParser();
		        $loginLogs = array_map(function ($log) use ($uaParser) {
							if ($log["useragent"] == null) {
								$log["browser"] = "Unknown";
								$log["os"] = "Unknown";
							} else {
								$ua = $uaParser->parse($log["useragent"]);
								$log["browser"] = $ua->browser() ?? 'Unknown';
								$log["os"] = $ua->platform() ?? 'Unknown';
							}
							
							return $log;
		        }, $loginLogs->fetchAll(PDO::FETCH_ASSOC));
						
						$tfaStatus = false;
						if (!moduleIsDisabled('tfa')) {
							$tfaKey = db()->prepare("SELECT * FROM AccountTfaKeys WHERE accountID = ?");
							$tfaKey->execute(array($account["id"]));
							$tfaKey = $tfaKey->fetch();
							if ($tfaKey) {
								$tfaStatus = true;
							}
						}
						
						$totalSpent = db()->prepare("SELECT SUM(earnings) as totalSpent FROM Orders WHERE accountID = ? AND status = ?");
						$totalSpent->execute(array($account["id"], 1));
						$account["totalSpent"] = $totalSpent->fetchColumn();
	        
		        $linkedAccounts = [];
		        if (DiscordBotService::isBotReady() || !moduleIsDisabled('account_linking')) {
			        $linkedAccounts = db()->prepare("SELECT P.id, P.name, P.slug, L.identifier, L.displayName FROM LinkedAccounts L INNER JOIN AccountLinkingProviders P ON P.id = L.providerID WHERE L.accountID = ?");
			        $linkedAccounts->execute([$account["id"]]);
			        $linkedAccounts = $linkedAccounts->fetchAll(PDO::FETCH_ASSOC);
		        }
		        $account["linkedAccounts"] = $linkedAccounts;

            return view('accounts.show', compact('account', 'lastSeen', 'lastSeen', 'accountCustomFieldValues', 'siteBannedAccountStatus', 'supportBannedAccountStatus', 'commentBannedAccountStatus', 'statServers', 'statServer', 'statServerTableItems', 'statServerData', 'statServerUser', 'statServerUserRank', 'chests', 'tickets', 'creditLogs', 'orders', 'lotteryLogs', 'giftLogs', 'chestLogs', 'bazaarLogs', 'applications', 'loginLogs', 'tfaStatus'));
        } else {
            return back();
        }
    }

    public function destroy($id)
    {
        // Prevent deleting your own account
        if (auth()->user()->id() == $id) {
            return back()->flash("error", t__('You cannot delete yourself!'));
        }

        $account = db()->prepare("SELECT * FROM Accounts WHERE id = ?");
        $account->execute([$id]);
        $account = $account->fetch();
        if (!$account) return view('404');

        // Security Check
        $account["permissions"] = getPermissions($account["id"]);
        if (!auth()->user()->isSuperAdmin() && checkPerm($account, 'SUPER_ADMIN')) {
          return back()->flash("error", t__('You cannot delete an account with a higher priority than your highest role!'));
        }

        // Security Check
        $accountRoles = getRoles($account["id"], true);
        if (!auth()->user()->isSuperAdmin() && count($accountRoles) > 0) {
          $isHigher = RoleService::checkHigherRoleByUserId(auth()->user()->id(), $accountRoles[0]["priority"]);
          if (!$isHigher) {
            return back()->flash("error", t__('You cannot delete an account with a higher priority than your highest role!'));
          }
        }

        $deleteAccount = db()->prepare("DELETE FROM Accounts WHERE id = ?");
        $deleteAccount->execute([$id]);
				
				$deleteAccountDiscordData = db()->prepare("DELETE FROM AccountDiscordData WHERE accountID = ?");
				$deleteAccountDiscordData->execute([$id]);
				
				$deleteLinkedAccounts = db()->prepare("DELETE FROM LinkedAccounts WHERE accountID = ?");
				$deleteLinkedAccounts->execute([$id]);
				
        return back();
    }

  public function unlinkDiscord($id)
  {
    $account = db()->prepare("SELECT * FROM Accounts WHERE id = ?");
    $account->execute([$id]);
    $account = $account->fetch();
    if (!$account) return view('404');

    $accountDiscordData = db()->prepare("SELECT * FROM AccountDiscordData WHERE accountID = ?");
    $accountDiscordData->execute(array($id));
    $accountDiscordData = $accountDiscordData->fetch();

    if (!$accountDiscordData)
      return back();

    // Security Check
    $account["permissions"] = getPermissions($account["id"]);
    if (!auth()->user()->isSuperAdmin() && checkPerm($account, 'SUPER_ADMIN')) {
      return back()->flash("error", t__('You cannot edit an account with a higher priority than your highest role!'));
    }

    // Security Check
    $accountRoles = getRoles($account["id"], true);
    if (!auth()->user()->isSuperAdmin() && count($accountRoles) > 0) {
      $isHigher = RoleService::checkHigherRoleByUserId(auth()->user()->id(), $accountRoles[0]["priority"]);
      if (!$isHigher) {
        return back()->flash("error", t__('You cannot edit an account with a higher priority than your highest role!'));
      }
    }

    $discord = new Discord(moduleSettings('discord', 'clientID'), moduleSettings('discord', 'clientSecret'), moduleSettings('discord', 'botToken'));
    $discord->removeRole(moduleSettings('discord', 'guildID'), $accountDiscordData["discordUserID"], moduleSettings('discord', 'syncedRoleID'));

    $unlinkDiscord = db()->prepare("DELETE FROM AccountDiscordData WHERE accountID = ?");
    $unlinkDiscord->execute([$accountDiscordData["accountID"]]);

    return back();
  }
	
	public function unlinkAccount($id, $provider)
	{
		$account = db()->prepare("SELECT * FROM Accounts WHERE id = ?");
		$account->execute([$id]);
		$account = $account->fetch();
		if (!$account) return view('404');
		
		$accountLinkingProvider = db()->prepare("SELECT * FROM AccountLinkingProviders WHERE slug = ?");
		$accountLinkingProvider->execute([$provider]);
		$accountLinkingProvider = $accountLinkingProvider->fetch();
		if (!$accountLinkingProvider) return view('404');
		
		$linkedAccount = db()->prepare("SELECT * FROM LinkedAccounts WHERE accountID = ? AND providerID = ?");
		$linkedAccount->execute([$account['id'], $accountLinkingProvider['id']]);
		$linkedAccount = $linkedAccount->fetch(PDO::FETCH_ASSOC);
		if (!$linkedAccount) return view('404');
		
		// Security Check
		$account["permissions"] = getPermissions($account["id"]);
		if (!auth()->user()->isSuperAdmin() && checkPerm($account, 'SUPER_ADMIN')) {
			return back()->flash("error", t__('You cannot edit an account with a higher priority than your highest role!'));
		}
		
		// Security Check
		$accountRoles = getRoles($account["id"], true);
		if (!auth()->user()->isSuperAdmin() && count($accountRoles) > 0) {
			$isHigher = RoleService::checkHigherRoleByUserId(auth()->user()->id(), $accountRoles[0]["priority"]);
			if (!$isHigher) {
				return back()->flash("error", t__('You cannot edit an account with a higher priority than your highest role!'));
			}
		}
		
		if ($accountLinkingProvider['slug'] == 'discord') {
			// Send request to LeaderOS Discord Bot API to remove discord roles
			$siteRoles = db()->query("SELECT discordRoleID FROM Roles WHERE discordRoleID IS NOT NULL AND discordRoleID != ''");
			$siteRoles = $siteRoles->fetchAll(PDO::FETCH_ASSOC);
			
			DiscordBotService::unlinkUser($linkedAccount['identifier'], array_column($siteRoles, 'discordRoleID'));
		}
		
		$unlinkAccount = db()->prepare("DELETE FROM LinkedAccounts WHERE accountID = ? AND providerID = ?");
		$unlinkAccount->execute([$account['id'], $accountLinkingProvider['id']]);
		
		return back();
	}
	
	public function removeTfa($id)
	{
		$account = db()->prepare("SELECT * FROM Accounts WHERE id = ?");
		$account->execute([$id]);
		$account = $account->fetch();
		if (!$account) return view('404');
		
		// Security Check
		$account["permissions"] = getPermissions($account["id"]);
		if (!auth()->user()->isSuperAdmin() && checkPerm($account, 'SUPER_ADMIN')) {
			return back()->flash("error", t__('You cannot edit an account with a higher priority than your highest role!'));
		}
		
		// Security Check
		$accountRoles = getRoles($account["id"], true);
		if (!auth()->user()->isSuperAdmin() && count($accountRoles) > 0) {
			$isHigher = RoleService::checkHigherRoleByUserId(auth()->user()->id(), $accountRoles[0]["priority"]);
			if (!$isHigher) {
				return back()->flash("error", t__('You cannot edit an account with a higher priority than your highest role!'));
			}
		}
		
		$deleteTfaKeys = db()->prepare("DELETE FROM AccountTfaKeys WHERE accountID = ?");
		$deleteTfaKeys->execute(array($id));
		
		return back();
	}
	
	public function uploadAvatar($id)
	{
		if (settings('allowCustomAvatars') == 0) return view('404');
		
		validate([
			'avatar' => 'required|uploaded_file|max:2M|mimes:jpeg,jpg,png,gif'
		]);
		
		$account = db()->prepare("SELECT * FROM Accounts WHERE id = ?");
		$account->execute([$id]);
		$account = $account->fetch();
		if (!$account) return view('404');
		
		// Security Check
		$account["permissions"] = getPermissions($account["id"]);
		if (!auth()->user()->isSuperAdmin() && checkPerm($account, 'SUPER_ADMIN')) {
			return back()->flash("error", t__('You cannot edit an account with a higher priority than your highest role!'));
		}
		
		// Security Check
		$accountRoles = getRoles($account["id"], true);
		if (!auth()->user()->isSuperAdmin() && count($accountRoles) > 0) {
			$isHigher = RoleService::checkHigherRoleByUserId(auth()->user()->id(), $accountRoles[0]["priority"]);
			if (!$isHigher) {
				return back()->flash("error", t__('You cannot edit an account with a higher priority than your highest role!'));
			}
		}
		
		$upload = new Upload($_FILES["avatar"]);
		$imageID = md5(uniqid(rand(0, 9999999)));
		if ($upload->uploaded) {
			$upload->allowed = ["image/png", "image/gif", "image/jpg", "image/jpeg"];
			$upload->file_new_name_body = $imageID;
			$upload->process(__ROOT__ . "/apps/main/public/images/avatars/");
			if ($upload->processed) {        
				$imageType = $upload->file_dst_name_ext;
				
				try {
          $imageSourcePath = __ROOT__ . "/apps/main/public/images/avatars/" . $imageID . "." . $imageType;
					ImageCropService::processAvatar($imageSourcePath, 256);
				} catch (Exception $e) {
					return back()->flash("error", t__('An error occupied while uploading an image: %error%', ['%error%' => $e->getMessage()]));
				}
				
				$currentAvatar = db()->prepare("SELECT * FROM Avatars WHERE accountID = ?");
				$currentAvatar->execute(array($id));
				$currentAvatar = $currentAvatar->fetch();
				if ($currentAvatar) {
					$updateAvatar = db()->prepare("UPDATE Avatars SET imageID = ?, imageType = ? WHERE accountID = ?");
					$updateAvatar->execute(array($imageID, $imageType, $id));
					
					// Delete the old avatar
					@unlink(__ROOT__ . "/apps/main/public/images/avatars/" . $currentAvatar["imageID"] . "." . $currentAvatar["imageType"]);
				} else {
					$insertAvatar = db()->prepare("INSERT INTO Avatars (accountID, imageID, imageType) VALUES (?, ?, ?)");
					$insertAvatar->execute(array($id, $imageID, $imageType));
				}
				
				return back()->flash("success", t__('Changes has been saved successfully!'));
			} else {
				return back()->flash("error", t__('An error occupied while uploading an image: %error%', ['%error%' => $upload->error]));
			}
		} else {
			return back()->flash("error", t__('An error occupied while uploading an image: %error%', ['%error%' => $upload->error]));
		}
	}
	
	public function removeAvatar($id)
	{
		$avatar = db()->prepare("SELECT * FROM Avatars WHERE accountID = ?");
		$avatar->execute(array($id));
		$avatar = $avatar->fetch();
		if (!$avatar) return back()->flash("error", t__('Not Found!'));
		
		$account = db()->prepare("SELECT * FROM Accounts WHERE id = ?");
		$account->execute([$id]);
		$account = $account->fetch();
		if (!$account) return view('404');
		
		// Security Check
		$account["permissions"] = getPermissions($account["id"]);
		if (!auth()->user()->isSuperAdmin() && checkPerm($account, 'SUPER_ADMIN')) {
			return back()->flash("error", t__('You cannot edit an account with a higher priority than your highest role!'));
		}
		
		// Security Check
		$accountRoles = getRoles($account["id"], true);
		if (!auth()->user()->isSuperAdmin() && count($accountRoles) > 0) {
			$isHigher = RoleService::checkHigherRoleByUserId(auth()->user()->id(), $accountRoles[0]["priority"]);
			if (!$isHigher) {
				return back()->flash("error", t__('You cannot edit an account with a higher priority than your highest role!'));
			}
		}
		
		$deleteAvatar = db()->prepare("DELETE FROM Avatars WHERE accountID = ?");
		$deleteAvatar->execute(array($id));
		
		// Delete Image
		@unlink(__ROOT__ . "/apps/main/public/images/avatars/" . $avatar["imageID"] . "." . $avatar["imageType"]);
		
		return back()->flash("success", t__('Changes has been saved successfully!'));
	}
}