<?php

namespace Dashboard\Controllers\Blog;

use Dashboard\Core\Controller;
use Dashboard\Core\Jobs\JobDispatcher;
use Dashboard\Core\Redirect;
use Dashboard\Jobs\SendDiscordWebhook;
use Dashboard\Services\SlugService;
use PDO;
use Verot\Upload\Upload;

class PostController extends Controller
{
    public function __construct()
    {
        abort_perm('MANAGE_BLOG');
    }

    public function index()
    {
        if (isset($_GET["page"])) {
            if (!is_numeric($_GET["page"])) {
                $_GET["page"] = 1;
            }
            $page = intval(get("page"));
        } else {
            $page = 1;
        }

        $visiblePageCount = 5;
        $limit = 50;

        $posts = db()->query("SELECT count(N.id) FROM News N INNER JOIN Accounts A ON N.accountID = A.id INNER JOIN NewsCategories NC ON N.categoryID = NC.id ORDER BY N.id DESC");
        $itemsCount = $posts->fetchColumn();
        $pageCount = ceil($itemsCount / $limit);
        if ($page > $pageCount) {
            $page = 1;
        }
        $visibleItemsCount = $page * $limit - $limit;
        $posts = db()->query("SELECT N.*, A.username, A.realname, NC.name as categoryName FROM News N INNER JOIN Accounts A ON N.accountID = A.id INNER JOIN NewsCategories NC ON N.categoryID = NC.id ORDER BY N.id DESC LIMIT $visibleItemsCount, $limit");

        if (isset($_GET["search"])) {
            if (get("search") != null) {
                $posts = db()->prepare("SELECT N.*, A.username, A.realname, NC.name as categoryName FROM News N INNER JOIN Accounts A ON N.accountID = A.id INNER JOIN NewsCategories NC ON N.categoryID = NC.id WHERE N.title LIKE :search OR (A.realname LIKE :search OR A.username LIKE :search) OR NC.name LIKE :search ORDER BY N.id DESC");
                $posts->execute(array(
                    "search" => '%' . get("search") . '%'
                ));
            }
        }
        $posts = array_map(function ($blogPost) {
            $postComments = db()->prepare("SELECT NC.id FROM NewsComments NC INNER JOIN Accounts A ON NC.accountID = A.id WHERE NC.newsID = ? AND NC.status = ?");
            $postComments->execute(array($blogPost["id"], 1));
            $blogPost["commentsCount"] = $postComments->rowCount();
            return $blogPost;
        }, $posts->fetchAll());

        return view('blog.posts.index', compact('posts', 'page', 'visiblePageCount', 'pageCount'));
    }

    public function create()
    {
        $categories = db()->query("SELECT * FROM NewsCategories");
        $categories = $categories->fetchAll();

        return view('blog.posts.create', compact('categories'));
    }

    public function store()
    {
				validate([
					'status' => 'required|in:published,draft',
				]);
				
	      if (input('status') == 'draft') {
		      return self::saveDraft();
				}
			
        validate([
            'title' => 'required',
            'categoryID' => 'required',
            'content' => 'required',
            'commentsStatus' => 'required',
            'image' => 'required|uploaded_file',
        ]);
		    
				$category = db()->prepare("SELECT * FROM NewsCategories WHERE id = ?");
				$category->execute([input("categoryID")]);
				$category = $category->fetch();
				if (!$category) return back()->flash("error", t__('Category not found!'));
				
        $upload = new Upload($_FILES["image"]);
        $imageID = md5(uniqid(rand(0, 9999)));
        if ($upload->uploaded) {
            $upload->allowed = array("image/*");
            $upload->file_new_name_body = $imageID;
            $upload->process(__ROOT__ . "/apps/main/public/images/blog/posts/");
            if ($upload->processed) {
								$slug = SlugService::preventDuplicate(input("title"), 'News');
                $insertPost = db()->prepare("INSERT INTO News (accountID, title, slug, categoryID, content, imageID, imageType, views, commentsStatus, status, updateDate, creationDate) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
		            $insertPost->execute(array(auth()->user()->id(), input("title"), $slug, input("categoryID"), filteredContent($_POST["content"]), $imageID, $upload->file_dst_name_ext, 0, input("commentsStatus"), "published", datetime(), datetime()));
                $postID = db()->lastInsertId();

                if (input("tags") != null) {
                    $tags = explode(',', trim(input("tags"), ','));
		                $insertTags = db()->prepare("INSERT INTO NewsTags (newsID, name, slug) VALUES (?, ?, ?)");
                    foreach ($tags as $tag) {
		                    $insertTags->execute(array($postID, $tag, slugify($tag)));
                    }
                }
	            
		            JobDispatcher::dispatch((new SendDiscordWebhook('blog.post.published', [
			            "username" => auth()->user()->displayName(),
			            "title" => input("title"),
			            "category" => $category["name"],
			            "content" => substr(strip_tags(input('content')), 0, 250) . "...",
			            "thumbnail" => websiteUrl('/assets/core/images/blog/posts/' . $imageID . '.' . $upload->file_dst_name_ext),
			            "url" => websiteUrl("/blog/posts/$slug")
		            ])));

                return back()->flash("success", t__('Post has been published successfully!'));
            }
        } else {
            return back()->flash("error", t__('An error occupied while uploading an image: %error%', ['%error%' => $upload->error]));
        }
    }

    public function edit($id)
    {
        $post = db()->prepare("SELECT * FROM News WHERE id = ?");
        $post->execute(array($id));
        $post = $post->fetch();

	      if (!$post) return view('404');

        $categories = db()->query("SELECT * FROM NewsCategories");
        $categories = $categories->fetchAll();

        $tags = db()->prepare("SELECT NT.name FROM NewsTags NT INNER JOIN News N ON NT.newsID = N.id WHERE NT.newsID = ?");
        $tags->execute(array($post["id"]));
        $post["tags"] = $tags->fetchAll(PDO::FETCH_COLUMN, 0);

        return view('blog.posts.edit', compact('post', 'categories'));
    }

    public function update($id)
    {
		    validate([
			    'status' => 'required|in:published,draft',
		    ]);
		    
		    if (input('status') == 'draft') {
			    return self::saveDraft($id);
		    }
				
        validate([
            'title' => 'required',
            'categoryID' => 'required',
            'content' => 'required',
            'commentsStatus' => 'required',
        ]);
	    
		    $post = db()->prepare("SELECT * FROM News WHERE id = ?");
		    $post->execute(array($id));
		    $post = $post->fetch();
		    
		    if (!$post) return view('404');
				
				// If switch from draft to published and image is not uploaded
		    if ($post["status"] == "draft" && input("status") == "published" && $post['imageID'] == "" && $post['imageType'] == "" && !input()->file('image')->size) {
			    return back()->flash("error", t__('Please upload an image before publishing the post!'));
		    }

        if (input()->file('image')->size) {
            $upload = new Upload($_FILES["image"]);
            $imageID = md5(uniqid(rand(0, 9999)));
            if ($upload->uploaded) {
                $upload->allowed = array("image/*");
                $upload->file_overwrite = true;
                $upload->file_new_name_body = $imageID;
                $upload->process(__ROOT__ . "/apps/main/public/images/blog/posts/");
                if ($upload->processed) {
                    $updateNews = db()->prepare("UPDATE News SET imageID = ?, imageType = ? WHERE id = ?");
                    $updateNews->execute(array($imageID, $upload->file_dst_name_ext, $id));
                } else {
                    return back()->flash("error", t__('An error occupied while uploading an image: %error%', ['%error%' => $upload->error]));
                }
            }
        }
	    
		    if ($post["title"] != input("title")) {
			    $slug = SlugService::preventDuplicate(input("title"), 'News');
		    } else {
			    $slug = $post["slug"];
		    }
				
        $updatePost = db()->prepare("UPDATE News SET title = ?, slug = ?, categoryID = ?, content = ?, commentsStatus = ?, status = ? WHERE id = ?");
		    $updatePost->execute(array(input("title"), $slug, input("categoryID"), filteredContent($_POST["content"]), input("commentsStatus"), "published", $id));

        $deleteTags = db()->prepare("DELETE FROM NewsTags WHERE newsID = ?");
		    $deleteTags->execute(array($id));
        if (input("tags") != null) {
            $tags = explode(',', trim(input("tags"), ','));
            $insertTags = db()->prepare("INSERT INTO NewsTags (newsID, name, slug) VALUES (?, ?, ?)");
            foreach ($tags as $tag) {
	            $insertTags->execute(array($id, $tag, slugify($tag)));
            }
        }

        return back()->flash("success", t__('Changes has been saved successfully!'));
    }

    public function destroy($id)
    {
        $deleteNews = db()->prepare("DELETE FROM News WHERE id = ?");
        $deleteNews->execute(array($id));
        return back();
    }
		
		private function saveDraft($id = null)
		{
			validate([
				'title' => 'required',
				'categoryID' => 'required',
				'content' => 'required',
			]);
			
			$title = input("title");
			$categoryID = input("categoryID");
			$content = filteredContent($_POST["content"]);
			$imageID = "";
			$imageExtension = "";
			$commentsStatus = input("commentsStatus") ?? 1;
			$status = "draft";
			
			$post = null;
			if ($id !== null) {
				$post = db()->prepare("SELECT * FROM News WHERE id = ?");
				$post->execute(array($id));
				$post = $post->fetch();
				
				if (!$post) return view('404');
			}
			
			$category = db()->prepare("SELECT * FROM NewsCategories WHERE id = ?");
			$category->execute([$categoryID]);
			$category = $category->fetch();
			if (!$category) return back()->flash("error", t__('Category not found!'));
			
			if (input()->file('image')->size) {
				$upload = new Upload($_FILES["image"]);
				$imageID = md5(uniqid(rand(0, 9999)));
				if ($upload->uploaded) {
					$upload->allowed = array("image/*");
					$upload->file_overwrite = true;
					$upload->file_new_name_body = $imageID;
					$upload->process(__ROOT__ . "/apps/main/public/images/blog/posts/");
					if ($upload->processed) {
						if ($post !== null) {
							$imageExtension = $upload->file_dst_name_ext;
							$updateNews = db()->prepare("UPDATE News SET imageID = ?, imageType = ? WHERE id = ?");
							$updateNews->execute(array($imageID, $imageExtension, $post["id"]));
						}
					} else {
						return back()->flash("error", t__('An error occupied while uploading an image: %error%', ['%error%' => $upload->error]));
					}
				}
			}
			
			if ($post !== null && $post["title"] == $title) {
				$slug = $post["slug"];
			} else {
				$slug = SlugService::preventDuplicate($title, 'News');
			}
			
			if ($post === null) {
				$insertPost = db()->prepare("INSERT INTO News (accountID, title, slug, categoryID, content, imageID, imageType, views, commentsStatus, status, updateDate, creationDate) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
				$insertPost->execute(array(auth()->user()->id(), $title, $slug, $categoryID, $content, $imageID, $imageExtension, 0, $commentsStatus, $status, datetime(), datetime()));
				$postID = db()->lastInsertId();
				
				if (input("tags") != null) {
					$tags = explode(',', trim(input("tags"), ','));
					$insertTags = db()->prepare("INSERT INTO NewsTags (newsID, name, slug) VALUES (?, ?, ?)");
					foreach ($tags as $tag) {
						$insertTags->execute(array($postID, $tag, slugify($tag)));
					}
				}
				
				$editPostRedirect = new Redirect(url('dashboard.blog.posts.edit', ['id' => $postID]));
				return $editPostRedirect->flash("success", t__('Draft saved successfully!'));
			} else {
				$updatePost = db()->prepare("UPDATE News SET title = ?, slug = ?, categoryID = ?, content = ?, commentsStatus = ?, status = ? WHERE id = ?");
				$updatePost->execute(array($title, $slug, $categoryID, $content, $commentsStatus, $status, $post["id"]));
				
				$deleteTags = db()->prepare("DELETE FROM NewsTags WHERE newsID = ?");
				$deleteTags->execute(array($post["id"]));
				if (input("tags") != null) {
					$tags = explode(',', trim(input("tags"), ','));
					$insertTags = db()->prepare("INSERT INTO NewsTags (newsID, name, slug) VALUES (?, ?, ?)");
					foreach ($tags as $tag) {
						$insertTags->execute(array($post["id"], $tag, slugify($tag)));
					}
				}
				
				return back()->flash('success', t__('Draft saved successfully!'));
			}
		}
}