<?php

namespace Dashboard\Controllers;

use Dashboard\Core\Controller;
use Dashboard\Services\RoleService;
use PDO;

class RoleController extends Controller
{
    public function __construct()
    {
        abort_perm('MANAGE_ROLES');
    }

    public function index()
    {
        $roles = db()->query("SELECT * FROM Roles ORDER BY priority DESC, id DESC");
        $roles = $roles->fetchAll(PDO::FETCH_ASSOC);

        return view('roles.index', compact('roles'));
    }

    public function create()
    {
        $permissions = db()->query("SELECT * FROM Permissions");
        $permissions = $permissions->fetchAll(PDO::FETCH_ASSOC);

        return view('roles.create', compact('permissions'));
    }

    public function store()
    {
        validate([
            'name' => 'required',
            'priority' => 'required'
        ]);

        // Security Check
        if (!auth()->user()->isSuperAdmin()) {
          $isHigher = RoleService::checkHigherRoleByUserId(auth()->user()->id(), input("priority"));
          if (!$isHigher) {
            return back()->flash("error", t__('You cannot create a role with a higher priority than your highest role!'));
          }
        }
	    
		    $slug = slugify(input("name"));

        $insertRole = db()->prepare("INSERT INTO Roles (name, slug, priority, usernameCSS, badgeCSS, discordRoleID) VALUES (?, ?, ?, ?, ?, ?)");
        $insertRole->execute(array(input("name"), $slug, input("priority"), input("usernameCSS"), input("badgeCSS"), input("discordRoleID")));
        $roleID = db()->lastInsertId();

        if (input("isStaffRole") == 1) {
            $addPermToUser = db()->prepare("INSERT INTO RolePermissions (roleID, permissionID) VALUES (?, ?)");
            if (isset($_POST["permissions"])) {
              // Security Check
              if (!auth()->user()->isSuperAdmin()) {
                $isDifferent = RoleService::isDifferentPermissions(auth()->user()->permissions(), $_POST["permissions"]);
                if ($isDifferent) {
                  return back()->flash("error", t__('You cannot select a permission you do not have!'));
                }
              }

              foreach ($_POST["permissions"] as $permission) {
                $permission = strip_tags($permission);
                $addPermToUser->execute(array($roleID, $permission));
              }
            }

            // Add VIEW_DASHBOARD permission
            $getViewDashboardRole = db()->prepare("SELECT id FROM Permissions WHERE name = ?");
            $getViewDashboardRole->execute(array("VIEW_DASHBOARD"));
            $getViewDashboardRole = $getViewDashboardRole->fetch(PDO::FETCH_ASSOC);
            $addPermToUser->execute(array($roleID, $getViewDashboardRole["id"]));
        }
        return back()->flash("success", t__('Role has been added successfully!'));
    }

    public function edit($id)
    {
        $role = db()->prepare("SELECT * FROM Roles WHERE id = ?");
        $role->execute(array($id));
        $role = $role->fetch();

        if (!$role) return view('404');

        // Security Check
        if (!auth()->user()->isSuperAdmin()) {
          $isHigher = RoleService::checkHigherRoleByUserId(auth()->user()->id(), $role["priority"]);
          if (!$isHigher) {
            abort_403(
              null,
              t__('You cannot edit a role with a higher priority than your highest role!')
            );
          }
        }

        $getViewDashboardPermission = db()->prepare("SELECT id FROM Permissions WHERE name = ?");
        $getViewDashboardPermission->execute(array("VIEW_DASHBOARD"));
        $getViewDashboardPermission = $getViewDashboardPermission->fetch(PDO::FETCH_ASSOC);

        $getSuperAdminPermission = db()->prepare("SELECT id FROM Permissions WHERE name = ?");
        $getSuperAdminPermission->execute(array("SUPER_ADMIN"));
        $getSuperAdminPermission = $getSuperAdminPermission->fetch(PDO::FETCH_ASSOC);

        $checkIsStaffRole = db()->prepare("SELECT * FROM RolePermissions WHERE roleID = ? AND permissionID IN (?, ?)");
        $checkIsStaffRole->execute(array($role["id"], $getViewDashboardPermission["id"], $getSuperAdminPermission["id"]));
        $isStaffRole = $checkIsStaffRole->rowCount() > 0;

        $permissions = db()->query("SELECT * FROM Permissions");
        $permissions = $permissions->fetchAll(PDO::FETCH_ASSOC);

        $rolePermissionList = [];
        $rolePermissions = db()->prepare("SELECT permissionID FROM RolePermissions WHERE roleID = ?");
        $rolePermissions->execute(array($role["id"]));
        foreach ($rolePermissions as $rolePermission) {
          $rolePermissionList[] = $rolePermission["permissionID"];
        }

        return view('roles.edit', compact('role', 'isStaffRole', 'permissions', 'rolePermissionList'));
    }

    public function update($id)
    {
        validate([
            'name' => 'required',
            'priority' => 'required'
        ]);

        $role = db()->prepare("SELECT * FROM Roles WHERE id = ?");
        $role->execute(array($id));
        $role = $role->fetch();

        if (!$role) return view('404');

        // Security Check
        if (!auth()->user()->isSuperAdmin()) {
          $isHigher = RoleService::checkHigherRoleByUserId(auth()->user()->id(), [$role["priority"], input("priority")]);
          if (!$isHigher) {
            return back()->flash("error", t__('You cannot edit a role with a higher priority than your highest role!'));
          }

          if (isset($_POST["permissions"])) {
            $isDifferent = RoleService::isDifferentPermissions(auth()->user()->permissions(), $_POST["permissions"]);
            if ($isDifferent) {
              return back()->flash("error", t__('You cannot select a permission you do not have!'));
            }
          }
        }

        $getViewDashboardRole = db()->prepare("SELECT id FROM Permissions WHERE name = ?");
        $getViewDashboardRole->execute(array("VIEW_DASHBOARD"));
        $getViewDashboardRole = $getViewDashboardRole->fetch(PDO::FETCH_ASSOC);
	    
		    // Prevent deleting super admin permission if only one super admin exists
		    $hasSuperAdminPermission = db()->prepare("SELECT * FROM RolePermissions WHERE roleID = ? AND permissionID = ?");
		    $hasSuperAdminPermission->execute([$role["id"], 1]);
				
				// Has super admin permission and is trying to remove it
		    if ($hasSuperAdminPermission->rowCount() > 0 && !in_array(1, $_POST["permissions"])) {
			    $superAdminCount = db()->prepare("SELECT COUNT(*) FROM RolePermissions RP INNER JOIN Roles R ON R.id = RP.roleID WHERE RP.permissionID = 1 AND RP.roleID != ?");
			    $superAdminCount->execute([$id]);
			    $superAdminCount = $superAdminCount->fetchColumn();
			    if ($superAdminCount == 0) {
				    return back()->flash("error", t__('There must be at least one super admin role!'));
			    }
		    }

        $slug = ($role["slug"] == 'default') ? 'default' : slugify(input("name"));
        $updateRoles = db()->prepare("UPDATE Roles SET name = ?, slug = ?, priority = ?, usernameCSS = ?, badgeCSS = ?, discordRoleID = ? WHERE id = ?");
        $updateRoles->execute(array(input("name"), $slug, input("priority"), input("usernameCSS"), input("badgeCSS"), input("discordRoleID"), $id));

        $removePermsFromRole = db()->prepare("DELETE FROM RolePermissions WHERE roleID = ?");
        $removePermsFromRole->execute(array($role["id"]));
        if ($role["id"] != 1 && input("isStaffRole") == 1) {
            $addPermToUser = db()->prepare("INSERT INTO RolePermissions (roleID, permissionID) VALUES (?, ?)");
            if (isset($_POST["permissions"])) {
                foreach ($_POST["permissions"] as $permission) {
                    $permission = strip_tags($permission);
                    $addPermToUser->execute(array($role["id"], $permission));
                }
            }
            $addPermToUser->execute(array($role["id"], $getViewDashboardRole["id"]));
        }
        return back()->flash("success", t__('Changes has been saved successfully!'));
    }

    public function destroy($id)
    {
        // Prevent deleting default role
        if ($id == 1) {
            return back()->flash("error", t__('You cannot delete the default role!'));
        }

        $role = db()->prepare("SELECT * FROM Roles WHERE id = ?");
        $role->execute([$id]);
        $role = $role->fetch();
        if (!$role) return view('404');

        // Security Check
        if (!auth()->user()->isSuperAdmin()) {
          $isHigher = RoleService::checkHigherRoleByUserId(auth()->user()->id(), $role["priority"]);
          if (!$isHigher) {
            return back()->flash("error", t__('You cannot delete a role with a higher priority than your highest role!'));
          }
        }
				
				// Prevent deleting super admin role if only one super admin exists
	      $hasSuperAdminPermission = db()->prepare("SELECT * FROM RolePermissions WHERE roleID = ? AND permissionID = ?");
				$hasSuperAdminPermission->execute([$role["id"], 1]);
				if ($hasSuperAdminPermission->rowCount() > 0) {
					$superAdminCount = db()->prepare("SELECT COUNT(*) FROM RolePermissions RP INNER JOIN Roles R ON R.id = RP.roleID WHERE RP.permissionID = 1 AND RP.roleID != ?");
					$superAdminCount->execute([$role["id"]]);
					$superAdminCount = $superAdminCount->fetchColumn();
					if ($superAdminCount == 0) {
						return back()->flash("error", t__('There must be at least one super admin role!'));
					}
				}

        $deleteRole = db()->prepare("DELETE FROM Roles WHERE id = ?");
        $deleteRole->execute(array($id));
				
				$deleteRolePermissions = db()->prepare("DELETE FROM RolePermissions WHERE roleID = ?");
				$deleteRolePermissions->execute(array($id));

        return back();
    }
}