<?php
	/* LeaderOS
	 * Paraşüt OAuth Class
	 * @author : benfiratkaya
	 * @copyright : https://firatkaya.net
	 */
	
	namespace Dashboard\Libs;
	class Parasut
	{
		private string $baseUrl = "https://api.parasut.com";
		private string $clientID;
		private string $clientSecret;
		private string $companyID;
		
		public function __construct($clientID, $clientSecret, $companyID)
		{
			$this->clientID = $clientID;
			$this->clientSecret = $clientSecret;
			$this->companyID = $companyID;
		}
		
		/**
		 * Get Access Token
		 * @param $code
		 * @param $redirectUri
		 * @return mixed
		 * @throws Exception
		 */
		public function getAccessToken($code, $redirectUri)
		{
			$params = [
				'client_id' => $this->clientID,
				'client_secret' => $this->clientSecret,
				'grant_type' => 'authorization_code',
				'code' => $code,
				'redirect_uri' => $redirectUri
			];
			return $this->request("/oauth/token", 'POST', $params);
		}
		
		/**
		 * Get Access Token with Password
		 * @param $username
		 * @param $password
		 * @return mixed
		 * @throws Exception
		 */
		public function getAccessTokenWithPassword($username, $password)
		{
			$params = [
				'grant_type' => 'password',
				'client_id' => $this->clientID,
				'client_secret' => $this->clientSecret,
				'username' => $username,
				'password' => $password,
				'redirect_uri' => 'urn:ietf:wg:oauth:2.0:oob'
			];
			return $this->request("/oauth/token", 'POST', $params);
		}
		
		/**
		 * Refresh Access Token
		 * @param $refreshToken
		 * @return mixed
		 * @throws Exception
		 */
		public function refreshAccessToken($refreshToken)
		{
			$params = [
				'client_id' => $this->clientID,
				'client_secret' => $this->clientSecret,
				'grant_type' => 'refresh_token',
				'refresh_token' => $refreshToken,
			];
			return $this->request("/oauth/token", 'POST', $params);
		}
		
		/**
		 * Create a Product
		 *
		 * @param $accessToken
		 * @param $name
		 * @return mixed
		 * @throws Exception
		 */
		public function createProduct($accessToken, $name)
		{
			return $this->request("/v4/$this->companyID/products", 'POST', [
				'data' => [
					'type' => 'products',
					'attributes' => [
						'name' => $name
					]
				]
			], $this->createAccessTokenHeader($accessToken));
		}
		
		/**
		 * Create a Customer
		 *
		 * @param $accessToken
		 * @param $name
		 * @param $email
		 * @return mixed
		 * @throws Exception
		 */
		public function createCustomer($accessToken, $name, $email)
		{
			return $this->request("/v4/$this->companyID/contacts", 'POST', [
				'data' => [
					'type' => 'contacts',
					'attributes' => [
						'email' => $email,
						'name' => $name,
						'contact_type' => 'person',
						'account_type' => 'customer'
					]
				]
			], $this->createAccessTokenHeader($accessToken));
		}
		
		/**
		 * Get Customer
		 *
		 * @param $accessToken
		 * @param $contactID
		 * @return mixed
		 * @throws Exception
		 */
		public function getCustomer($accessToken, $contactID)
		{
			return $this->request("/v4/$this->companyID/contacts/$contactID", 'GET', [], $this->createAccessTokenHeader($accessToken));
		}
		
		/**
		 * Create a Invoice
		 *
		 * @param $accessToken
		 * @param $contactID
		 * @param $orderID
		 * @param $productID
		 * @param $orderTotal
		 * @return mixed
		 * @throws Exception
		 */
		public function createInvoice($accessToken, $contactID, $orderID, $productID, $orderTotal)
		{
			return $this->request("/v4/$this->companyID/sales_invoices", 'POST', [
				'data' => [
					'type' => 'sales_invoices',
					'attributes' => [
						'item_type' => 'invoice',
						'issue_date' => date('Y-m-d'),
						'due_date' => date('Y-m-d'),
						'invoice_id' => $orderID,
						'currency' => 'TRL',
					],
					"relationships" => [
						"details" => [
							"data" => [
								0 => [
									"type" => "sales_invoice_details",
									"attributes" => [
										"quantity" => 1,
										"unit_price" => $orderTotal / 1.20,
										"vat_rate" => 20,
										"description" => "Sipariş #$orderID"
									],
									"relationships" => [
										"product" => [
											"data" => [
												"id" => $productID,
												"type" => "products"
											]
										]
									]
								]
							]
						],
						"contact" => [
							"data" => [
								"type" => "contacts",
								"id" => $contactID
							]
						]
					]
				]
			], $this->createAccessTokenHeader($accessToken));
		}
		
		/**
		 * Pay Invoice
		 *
		 * @param $accessToken
		 * @param $invoiceID
		 * @param $accountID
		 * @param $amount
		 * @return mixed
		 * @throws Exception
		 */
		public function payInvoice($accessToken, $invoiceID, $accountID, $amount)
		{
			return $this->request("/v4/$this->companyID/sales_invoices/$invoiceID/payments", 'POST', [
				'data' => [
					'type' => 'payments',
					'attributes' => [
						'account_id' => $accountID,
						'date' => date('Y-m-d'),
						'amount' => $amount,
					]
				]
			], $this->createAccessTokenHeader($accessToken));
		}
		
		/**
		 * Create E-Archive
		 *
		 * @param $accessToken
		 * @param $invoiceID
		 * @param $paymentGateway
		 * @return mixed
		 * @throws Exception
		 */
		public function createEArchive($accessToken, $invoiceID, $paymentGateway)
		{
			return $this->request("/v4/$this->companyID/sales_invoices/$invoiceID/payments", 'POST', [
				'data' => [
					'type' => 'e_archives',
					'attributes' => [
						'internet_sale' => [
							'payment_type' => 'KREDIKARTI/BANKAKARTI',
							'payment_platform' => $paymentGateway,
							'payment_date' => date('Y-m-d'),
						],
					],
					'relationships' => [
						'sales_invoice' => [
							'data' => [
								'id' => $invoiceID,
								'type' => "sales_invoices"
							]
						]
					]
				]
			], $this->createAccessTokenHeader($accessToken));
		}
		
		/**
		 * Request
		 *
		 * @param string $url URL to fetch
		 * @param string $method HTTP method
		 * @param array $data
		 * @param array $headers Headers to send
		 * @return mixed
		 * @throws Exception
		 */
		public function request(string $url, string $method = 'GET', array $data = [], array $headers = [])
		{
			$url = $this->baseUrl . $url;
			$request = curl_init($url);
			
			// Curl options
			curl_setopt($request, CURLOPT_RETURNTRANSFER, true);
			curl_setopt($request, CURLOPT_CUSTOMREQUEST, $method);
			
			// Custom headers
			if (!empty($headers)) {
				curl_setopt($request, CURLOPT_HTTPHEADER, $this->createHeaders($headers));
				
				foreach ($headers as $key => $value) {
					if ($key === 'Content-Type' && $value === 'application/json') {
						$data = json_encode($data);
						break;
					}
				}
			}
			
			// Custom Data
			curl_setopt($request, CURLOPT_POSTFIELDS, $data);
			
			$response = curl_exec($request);
			$error = curl_error($request);
			$errorNo = curl_errno($request);
			curl_close($request);
			
			if ($errorNo !== 0)
				throw new Exception($error, $errorNo);
			
			return json_decode($response);
		}
		
		private function createAccessTokenHeader($token): array
		{
			return [
				'Content-Type' => 'application/json',
				'Authorization' => "Bearer " . $token
			];
		}
		
		// Creates curl headers from an array
		private function createHeaders($headers): array
		{
			$result = [];
			foreach ($headers as $key => $value) {
				$result[] = $key . ': ' . $value;
			}
			return $result;
		}
	}