<?php
	
	namespace Dashboard\Services;
	
	use Curl\Curl;
	use CURLFile;
	use Exception;
	
	class AiService
	{
		const API_BASE_URL = "https://ai-service-api.leaderos.net";
		
		/**
		 * @throws Exception
		 */
		public static function getSubscription()
		{
			$curl = new Curl();
			$curl->setDefaultJsonDecoder(true);
			$curl->setHeader('Content-Type', 'application/json');
			$curl->setHeader('Accept', 'application/json');
			$curl->setHeader('X-Api-Key', settings('apiKey'));
			$curl->get(self::API_BASE_URL . "/subscriptions");

			if ($curl->error || !isset($curl->response['status'])) {
				throw new Exception($curl->errorMessage);
			}
			
			if ($curl->response['status'] === false) {
				return null;
			}
			
			return $curl->response;
		}
		
		/**
		 * @throws Exception
		 */
		public static function generateWithAI($type, $prompt): string
		{
			global $templates;
			$lang = $templates->getData()['lang'] ?? settings('language');
			
			$curl = new Curl();
			$curl->setDefaultJsonDecoder(true);
			$curl->setHeader('Content-Type', 'application/json');
			$curl->setHeader('Accept', 'application/json');
			$curl->setHeader('X-Api-Key', settings('apiKey'));
			$curl->post(self::API_BASE_URL . "/generate-with-ai", [
				"type" => $type,
				"prompt" => $prompt,
				"game" => settings('gameType'),
				"serverName" => settings('serverName'),
				"language" => $lang,
			]);
			
			if ($curl->error) {
				throw new Exception($curl->errorMessage);
			}
			if (isset($curl->response['error'])) {
				if ($curl->response['error'] === 'NO_ACTIVE_SUBSCRIPTION') {
					throw new Exception(t__('You do not have an active AI subscription. Please subscribe to use AI features.'));
				}
				if ($curl->response['error'] === 'LIMIT_REACHED') {
					throw new Exception(t__('You have reached your AI usage limit for this month. Please wait until the next month or upgrade your subscription.'));
				}
				throw new Exception("Error from API: " . $curl->response['error']);
			}
			if (!isset($curl->response['status'])) {
				throw new Exception("Unexpected response from API: " . json_encode($curl->response));
			}
			if ($curl->response['status'] !== true) {
				throw new Exception("Failed to generate content: " . json_encode($curl->response));
			}
			if (!isset($curl->response['content'])) {
				throw new Exception("No content returned from API");
			}
			
			return $curl->response['content'];
		}
		
		/**
		 * @throws Exception
		 */
		public static function uploadFile($file)
		{
			$curl = new Curl();
			$curl->setDefaultJsonDecoder(true);
			$curl->setHeader('Content-Type', 'multipart/form-data');
			$curl->setHeader('Accept', 'application/json');
			$curl->setHeader('X-Api-Key', settings('apiKey'));
			$curl->post(self::API_BASE_URL . "/data-source/files", [
				"file" => new CURLFile($file['tmp_name'], $file['type'], $file['name']),
			]);
			
			if ($curl->error) {
				throw new Exception("Error uploading file: " . $curl->errorMessage);
			}
			if (isset($curl->response['error'])) {
				if ($curl->response['error'] === 'NO_ACTIVE_SUBSCRIPTION') {
					throw new Exception(t__('You do not have an active AI subscription. Please subscribe to use AI features.'));
				}
				if ($curl->response['error'] === 'LIMIT_REACHED') {
					throw new Exception(t__('You have reached your AI usage limit for this month. Please wait until the next month or upgrade your subscription.'));
				}
				throw new Exception("Error from API: " . $curl->response['error']);
			}
			if (!isset($curl->response['status'])) {
				throw new Exception("Unexpected response from API: " . json_encode($curl->response));
			}
			if ($curl->response['status'] !== true) {
				throw new Exception("Failed to upload file: " . json_encode($curl->response));
			}
			
			return true;
		}
		
		/**
		 * @throws Exception
		 */
		public static function addSitemap($url)
		{
			$curl = new Curl();
			$curl->setDefaultJsonDecoder(true);
			$curl->setHeader('Content-Type', 'application/json');
			$curl->setHeader('Accept', 'application/json');
			$curl->setHeader('X-Api-Key', settings('apiKey'));
			$curl->post(self::API_BASE_URL . "/data-source/sitemaps", [
				"url" => $url,
			]);
			
			if ($curl->error) {
				throw new Exception("Error adding sitemap: " . $curl->errorMessage);
			}
			if (isset($curl->response['error'])) {
				if ($curl->response['error'] === 'NO_ACTIVE_SUBSCRIPTION') {
					throw new Exception(t__('You do not have an active AI subscription. Please subscribe to use AI features.'));
				}
				if ($curl->response['error'] === 'LIMIT_REACHED') {
					throw new Exception(t__('You have reached your AI usage limit for this month. Please wait until the next month or upgrade your subscription.'));
				}
				throw new Exception("Error from API: " . $curl->response['error']);
			}
			if (!isset($curl->response['status'])) {
				throw new Exception("Unexpected response from API: " . json_encode($curl->response));
			}
			if ($curl->response['status'] !== true) {
				throw new Exception("Failed to add sitemap: " . json_encode($curl->response));
			}
			
			return true;
		}
		
		/**
		 * @throws Exception
		 */
		public static function deleteDataSource($id)
		{
			$curl = new Curl();
			$curl->setDefaultJsonDecoder(true);
			$curl->setHeader('Content-Type', 'application/json');
			$curl->setHeader('Accept', 'application/json');
			$curl->setHeader('X-Api-Key', settings('apiKey'));
			$curl->post(self::API_BASE_URL . "/delete-data-source", [
				"id" => $id,
			]);
			
			if ($curl->error) {
				throw new Exception("Error deleting data source: " . $curl->errorMessage);
			}
			if (isset($curl->response['error'])) {
				throw new Exception($curl->response['error']);
			}
			if (!isset($curl->response['status'])) {
				throw new Exception("Unexpected response from API: " . json_encode($curl->response));
			}
			if ($curl->response['status'] !== true) {
				throw new Exception("Failed to delete data source: " . json_encode($curl->response));
			}
			
			return true;
		}
	}