<?php
	namespace Dashboard\Core;
	
	class User {
		private int $id;
		private string $username;
		private string $displayName;
		private ?string $email;
		private string $avatar;
		private array $roles;
		private array $permissions;
		private array $supportPermissions;
		private bool $isTfaRequired;
		private array $data;
		
		public function __construct(
			int $id,
			string $username,
			string $displayName,
			?string $email,
			string $avatar,
			bool $isTfaVerified = null,
			array $roles = [],
			array $permissions = [],
			array $supportPermissions = [],
			array $data = []
		) {
			$this->id = $id;
			$this->username = $username;
			$this->displayName = $displayName;
			$this->email = $email;
			$this->avatar = $avatar;
			if (moduleIsDisabled('tfa')) {
				$this->isTfaRequired = false;
			} else {
				$this->isTfaRequired = $isTfaVerified === null ? false : !$isTfaVerified;
			}
			$this->roles = $roles;
			$this->permissions = $permissions;
			$this->data = $data;
			
			if ($this->can('MANAGE_SUPPORT')) {
				$supportPermissions['STRICT_MANAGE_SUPPORT_TICKETS'] = true;
			}
			$this->supportPermissions = $supportPermissions;
		}
		
		public function id(): int
		{
			return $this->id;
		}
		
		public function username(): string
		{
			return $this->username;
		}
		
		public function displayName(): string
		{
			return $this->displayName;
		}
		
		public function email(): ?string
		{
			return $this->email;
		}
		
		public function avatar(): string
		{
			return $this->avatar;
		}
		
		public function roles(): array
		{
			return $this->roles;
		}
		
		public function permissions(): array
		{
			return $this->permissions;
		}
		
		public function supportPermissions($permission = null)
		{
			return $permission === null ? $this->supportPermissions : $this->supportPermissions[$permission];
		}
		
		public function data(string $key)
		{
			return $this->data[$key];
		}
		
		public function can($perm): bool
		{
      if (is_array($perm)) {
        foreach ($perm as $p) {
          if ($this->can($p))
            return true;
        }
        return false;
      }

			if (in_array($perm, $this->permissions) || in_array('SUPER_ADMIN', $this->permissions))
				return true;
			
			return false;
		}
		
		public function isStaff(): bool
		{
			if ($this->can('VIEW_DASHBOARD'))
				return true;
			
			return false;
		}

    public function isSuperAdmin(): bool
    {
      if (in_array('SUPER_ADMIN', $this->permissions))
        return true;

      return false;
    }
		
		public function canManageSupportCategory($permission, $categoryId): bool
		{
			if ($this->supportPermissions['STRICT_MANAGE_SUPPORT_TICKETS'])
				return true;
			
			if (in_array($categoryId, $this->supportPermissions[$permission]))
				return true;
			
			return false;
		}
		
		public function isTfaRequired(): bool
		{
			return $this->isTfaRequired;
		}
	}