<?php
	
	use Cocur\Slugify\Slugify;
	use Dashboard\Core\Auth;
	use Dashboard\Core\Redirect;
  use League\CommonMark\GithubFlavoredMarkdownConverter;
  use Pecee\Http\Input\InputItem;
  use Pecee\SimpleRouter\SimpleRouter as Router;
	use Pecee\Http\Url;
	use Pecee\Http\Response;
	use Pecee\Http\Request;
	use Rakit\Validation\Validator;
	use Tamtamchik\SimpleFlash\Flash;

	/**
	 * Get url for a route by using either name/alias, class or method name.
	 *
	 * The name parameter supports the following values:
	 * - Route name
	 * - Controller/resource name (with or without method)
	 * - Controller class name
	 *
	 * When searching for controller/resource by name, you can use this syntax "route.name@method".
	 * You can also use the same syntax when searching for a specific controller-class "MyController@home".
	 * If no arguments is specified, it will return the url for the current loaded route.
	 *
	 * @param string|null $name
	 * @param string|array|null $parameters
	 * @param array|null $getParams
	 * @return \Pecee\Http\Url
	 * @throws \InvalidArgumentException
	 */
	function url(?string $name = null, $parameters = null, ?array $getParams = null): Url
	{
		return Router::getUrl($name, $parameters, $getParams);
	}
	
	/**
	 * @return \Pecee\Http\Response
	 */
	function response(): Response
	{
		return Router::response();
	}
	
	/**
	 * @return \Pecee\Http\Request
	 */
	function request(): Request
	{
		return Router::request();
	}
	
	/**
	 * Get input class
	 * @param string|null $index Parameter index name
	 * @param string|mixed|null $defaultValue Default return value
	 * @param array ...$methods Default methods
	 * @return \Pecee\Http\Input\InputHandler|array|string|null
	 */
	function input($index = null, $defaultValue = null, ...$methods)
	{
		if ($index !== null) {
			return request()->getInputHandler()->value($index, $defaultValue, ...$methods);
		}
		
		return request()->getInputHandler();
	}

  /**
   * Set input value
   * @param string $key
   * @param mixed $value
   */
  function set_input(string $key, $value): void
  {
    request()->getInputHandler()->addPost($key, new InputItem($key, $value));
  }
	
	/**
	 * @param string $url
	 * @param int|null $code
	 */
	function redirect(string $url, ?int $code = null): void
	{
		if ($url == "/login") {
			$url .= "?redirect=".$_SERVER["REQUEST_URI"];
		}
		
		if ($code !== null) {
			response()->httpCode($code);
		}
		
		response()->redirect($url);
	}
	
	/**
	 * Redirect back to previous page
	 * @return Redirect
	 */
	function back(): Redirect
	{
		return new Redirect(request()->getReferer());
	}
	
	/**
	 * Validate data
	 * @param array $data
	 * @param bool $back
	 * @return array
	 */
	function validate(array $data = [], bool $back = true): array {
		$validator = new Validator;
		
		$validation = $validator->validate($_POST + $_FILES, $data);
		if ($validation->fails()) {
			if ($back) {
				back()->flash("error", $validation->errors()->all());
			}
			return $validation->errors()->all();
		}
		return [];
	}
	
	/**
	 * Render view
	 * @param string $name
	 * @param array $data
	 * @return string
	 */
	function view(string $name, array $data = []): string
	{
		global $templates;
		return $templates->render('pages/'.str_replace('.', '/', $name), $data);
	}
	
	function auth(): Auth
	{
		return new Auth();
	}
	
	function user()
	{
		global $readAdmin;
		return $readAdmin;
	}
	
	/**
	 * Get Database instance
	 * @return PDO
	 */
	function db(): PDO
	{
		global $db;
		return $db;
	}

  /**
   * Get markdown instance
   * @return GithubFlavoredMarkdownConverter
   */
  function markdown(): GithubFlavoredMarkdownConverter
  {
	  static $instance;
	  if (is_null($instance)) {
		  $instance = new GithubFlavoredMarkdownConverter([
			  'html_input' => 'strip',
			  'allow_unsafe_links' => false,
		  ]);
	  }
	  
	  return $instance;
  }
	
	/**
	 * Get settings
	 * @param string $key
	 * @param string|null $default
	 * @return string|null
	 */
	function settings(string $key, string $default = null): ?string
	{
		global $readSettings;
		if (isset($readSettings[$key])) {
			return $readSettings[$key];
		}
		
		return $default;
	}

/**
 * Get module settings
 * @param string $module_slug
 * @param string $key
 * @param string|null $default
 * @return string|null
 */
function moduleSettings(string $module_slug, string $key, string $default = null): ?string
{
    static $settings = null;

    if (is_null($settings)) {
        $modules = db()->query("SELECT * FROM Modules");
        $modules = $modules->fetchAll(PDO::FETCH_ASSOC);

        $moduleSettings = db()->query("SELECT * FROM ModuleSettings");
        $moduleSettings = $moduleSettings->fetchAll(PDO::FETCH_ASSOC);
        
        // create settings array 'module_slug' => ['name' => 'value']
        $settings = [];
        foreach ($modules as $module) {
            $settings[$module['slug']] = [];
            foreach ($moduleSettings as $setting) {
                if ($setting['moduleID'] == $module['id']) {
                    $settings[$module['slug']][$setting['name']] = $setting['value'];
                }
            }
        }
    }

    return $settings[$module_slug][$key] ?? $default;
}

/**
 * Abort request and render view with code
 * @param int $code
 * @param array $data
 */
	function abort(int $code, array $data = []) {
		echo view($code, $data);

		exit();
	}

/**
 * Abort request with 403 code
 * @param string|null $title
 * @param string|null $message
 */
  function abort_403(string $title = null, string $message = null) {
    abort(403, [
      'title' => $title,
      'message' => $message
    ]);
  }
	
	/**
	 * Abort request if bool equals true
	 * @param bool $bool
	 * @param int $code
	 */
	function abort_if(bool $bool, int $code) {
		if ($bool) abort($code);
	}
	
	/**
	 * Abort request if bool equals false
	 * @param bool $bool
	 * @param int $code
	 */
	function abort_unless(bool $bool, int $code) {
		if (!$bool) abort($code);
	}
	
	/**
	 * Abort request if perm not exist
	 * @param string|array $perm
	 */
	function abort_perm($perm) {
		abort_unless(auth()->user()->can($perm), 403);
	}
	
	/**
	 * Get flash message
	 * @param string|null $type
	 */
	function _message(?string $type = null): void
	{
		echo Flash::display($type);
	}
	
	/**
	 * Echo old input value
	 */
	function _old(string $input, $default = null, $escape = true)
	{
		echo $escape ? htmlspecialchars(old($input, $default)) : old($input, $default);
	}

  /**
   * Get old input value
   */
  function old(string $input, $default = null)
  {
    if (isset($_SESSION['old'][$input]))
      return $_SESSION['old'][$input];

    return $default;
  }
	
	/**
	 * Get current csrf-token
	 * @return string|null
	 */
	function csrf_token(): ?string
	{
		$baseVerifier = Router::router()->getCsrfVerifier();
		if ($baseVerifier !== null) {
			return strip_tags($baseVerifier->getTokenProvider()->getToken());
		}
		
		return null;
	}
	
	/**
	 * Print csrf input field
	 * @return void
	 */
	function _csrf(): void
	{
		echo '<input type="hidden" name="csrf_token" value="' . csrf_token() . '">';
	}
	
	/**
	 * Print method input field
	 * @param string $method
	 * @return void
	 */
	function _method(string $method = "GET" | "POST" | "PUT" | "PATCH" | "DELETE"): void
	{
		echo '<input type="hidden" name="_method" value="' . $method . '">';
	}
	
	/**
	 * Selected if value is true
	 * @param bool $bool
	 * @return void
	 */
	function _selected(bool $bool): void
	{
		if ($bool) echo 'selected';
	}
	
	/**
	 * Checked if value is true
	 * @param bool|null $bool
	 * @return void
	 */
	function _checked(?bool $bool): void
	{
		if ($bool) echo 'checked';
	}
	
	/**
	 * Disabled if value is true
	 * @param bool $bool
	 * @return void
	 */
	function _disabled(bool $bool): void
	{
		if ($bool) echo 'disabled';
	}
	
	/**
	 * Active if value is true
	 * @param bool $bool
	 * @return void
	 */
	function _active(bool $bool): void
	{
		if ($bool) echo 'active';
	}
	
	function session($key, $value = null) {
		if ($value == null) {
			return $_SESSION[$key];
		}
		
		$_SESSION[$key] = $value;
		return $value;
	}
	
	function slugify($value): string
	{
		static $slugify = null;
		if ($slugify == null) {
			$slugify = new Slugify([
				'rulesets' => ['default', 'turkish', 'russian', 'hindi', 'chinese'],
			]);
		}
		$slugify->addRule("+", " plus ");
		$slugify->addRule("&", " and ");
		$slug = $slugify->slugify($value);
		if (trim($slug) == "") {
			return uniqid();
		}
		return $slug;
	}
	
	function go($url) {
		if ($url == "/login") {
			$url .= "?redirect=".$_SERVER["REQUEST_URI"];
		}
		header("Location: $url");
		exit();
	}
	
	function isRedirectable($url) {
		return preg_match("/^(\/)[a-zA-Z0-9_\-]+(\/|$)/", $url);
	}
	
	function goDelay($url, $wait = 0) {
		return '<script type="text/javascript">setTimeout("location.href=\''.$url.'\';", '.($wait*1000).');</script>';
	}
	
	function alertSuccess($text, $padding = true) {
		if ($padding == false) {
			return '<div class="alert alert-success mb-0"><i class="fa fa-check-circle d-none d-md-inline-block"></i> '.$text.'</div>';
		}
		else {
			return '<div class="alert alert-success"><i class="fa fa-check-circle d-none d-md-inline-block"></i> '.$text.'</div>';
		}
	}
	
	function alertError($text, $padding = true) {
		if ($padding == false) {
			return '<div class="alert alert-danger mb-0"><i class="fa fa-times-circle d-none d-md-inline-block"></i> '.$text.'</div>';
		}
		else {
			return '<div class="alert alert-danger"><i class="fa fa-times-circle d-none d-md-inline-block"></i> '.$text.'</div>';
		}
	}
	
	function alertWarning($text, $padding = true) {
		if ($padding == false) {
			return '<div class="alert alert-warning mb-0"><i class="fa fa-bell d-none d-md-inline-block"></i> '.$text.'</div>';
		}
		else {
			return '<div class="alert alert-warning"><i class="fa fa-bell d-none d-md-inline-block"></i> '.$text.'</div>';
		}
	}

  function avatar($username, $size = 20, $extraClass = null): string
  {
    return minecraftHead(settings('avatarAPI'), $username, $size, $extraClass);
  }
	
	function minecraftHead($avatarAPI = 1, $username = null, $size = 20, $extraClass = null) {
		if ($avatarAPI == 1) {
			$apiURL = "https://minotar.net/helm/$username/$size.png";
		}
		else if ($avatarAPI == 2) {
			$apiURL = "https://cravatar.eu/helmavatar/$username/$size.png";
		}
		else {
			$apiURL = "https://minotar.net/helm/$username/$size.png";
		}
		return '<img class="avatar lazyload '.$extraClass.'" data-src="'.$apiURL.'" src="/apps/main/public/images/loaders/head.png" alt="Oyuncu - '.$username.'" width="'.$size.'" height="'.$size.'">';
	}
	
	function minecraftHead2($avatarAPI = 1, $username = null, $size = 20, $extraClass) {
		if ($avatarAPI == 1) {
			$apiURL = "https://minotar.net/helm/$username/$size.png";
		}
		else if ($avatarAPI == 2) {
			$apiURL = "https://cravatar.eu/helmavatar/$username/$size.png";
		}
		else {
			$apiURL = "https://minotar.net/helm/$username/$size.png";
		}
		return '<img class="lazyload '.$extraClass.'" data-src="'.$apiURL.'" src="/apps/main/public/images/loaders/head.png" alt="Oyuncu - '.$username.'" width="'.$size.'" height="'.$size.'">';
	}
	
	function showEmoji($text) {
		$emojiPath = "/apps/main/public/images/emojis";
		$emojiText 	= array(
			":D",
			";)",
			":)",
			"<3",
			":(",
			":O",
			":o",
			":P",
			":')",
			":8",
			"-_-",
			"(y)"
		);
		$emojiImage  = array(
			'<img src="'.$emojiPath.'/1.png" width="18px" />',
			'<img src="'.$emojiPath.'/2.png" width="18px" />',
			'<img src="'.$emojiPath.'/3.png" width="18px" />',
			'<img src="'.$emojiPath.'/4.png" width="18px" />',
			'<img src="'.$emojiPath.'/5.png" width="18px" />',
			'<img src="'.$emojiPath.'/6.png" width="18px" />',
			'<img src="'.$emojiPath.'/6.png" width="18px" />',
			'<img src="'.$emojiPath.'/7.png" width="18px" />',
			'<img src="'.$emojiPath.'/8.png" width="18px" />',
			'<img src="'.$emojiPath.'/9.png" width="18px" />',
			'<img src="'.$emojiPath.'/10.png" width="18px" />',
			'<img src="'.$emojiPath.'/11.png" width="18px" />'
		);
		return str_ireplace($emojiText, $emojiImage, $text);
	}
	
	function post($parameter) {
		if (isset($_POST[$parameter])) {
			return htmlspecialchars(trim(strip_tags($_POST[$parameter])));
		}
		else {
			return false;
		}
	}
	
	function get($parameter) {
		if (isset($_GET[$parameter])) {
			return strip_tags(trim(addslashes($_GET[$parameter])));
		}
		else {
			return false;
		}
	}
	
	function filteredContent($content) {
		$contentBadHTMLTags = array('<script>', '</script>');
		return str_replace($contentBadHTMLTags, '', $content);
	}
	
	function limitedContent($content, $limit = 0) {
		$newsContentLength = strlen($content);
		if ($newsContentLength > $limit) {
			return mb_substr($content, 0, $limit, 'utf-8').'...';
		}
		else {
			return $content;
		}
	}
	
	function rememberPost($key, $default = "") {
		if (isset($_POST[$key])) return $_POST[$key];
		return $default;
	}
	
	function generateSalt($length) {
		$characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
		$randomString = '';
		for ($i = 0; $i < $length; $i++) {
			$randomString .= $characters[rand(0, strlen($characters) - 1)];
		}
		return $randomString;
	}
	
	function createSHA256($password){
		$salt = generateSalt(16);
		$hash = '$SHA$'.$salt.'$'.hash('sha256', hash('sha256', $password).$salt);
		return $hash;
	}
	
	function checkSHA256($password, $realPassword){
		$parts = explode('$', $realPassword);
		$salt = $parts[2];
		$hash = hash('sha256', hash('sha256', $password).$salt);
		$hash = '$SHA$'.$salt.'$'.$hash;
		return (($hash == $realPassword) ? true : false);
	}
	
	function validate_username($username): bool
	{
		$regex = settings('usernameRegex');
		return preg_match("/$regex/", $username) === 1;
	}
	
	function checkBadUsername($username = null, $list = array()) {
		foreach ($list as $badWord) {
			if (stristr($username, $badWord)) {
				return true;
			}
		}
		return false;
	}
	
	function checkEmail($email) {
		if (filter_var($email, FILTER_VALIDATE_EMAIL)) {
			/*$mailDomainWhitelist  = array(
				"yandex.com",
				"gmail.com",
				"hotmail.com",
				"hotmail.com.tr",
				"outlook.com",
				"outlook.com.tr",
				"aol.com",
				"icloud.com",
				"yahoo.com",
				"live.com",
				"mynet.com"
			);
			$mailExplode          = explode("@", $email);
			$mailDomain           = strtolower($mailExplode[1]);
			if (in_array($mailDomain, $mailDomainWhitelist)) {
				return false;
			}
			else {
				return true;
			}*/
			return false;
		}
		else {
			return true;
		}
	}
	
	function checkBadPassword($password) {
		$badPasswordList = array(
			'1234',
			'12345',
			'123456',
			'1234567',
			'12345678',
			'123456789',
			'1234567890',
			'abc123',
			'xyz123',
			'qwerty',
			'qwerty123',
			'sifre',
			'sifre0',
			'sifre123',
			'password',
			'password0'
		);
		return in_array($password, $badPasswordList);
	}
	
	function createCookie($name = null, $value = null, $duration = 0, $sslStatus = false) {
		setcookie($name, $value, (time()+($duration * 86400)), '/', '', $sslStatus, true);
		return true;
	}
	
	function removeCookie($name = null) {
		if (isset($_COOKIE[$name])) {
			setcookie($name, "", (time()-(999 * 86400)), '/');
			return true;
		}
		else {
			return false;
		}
	}

  function datetime() {
    return date("Y-m-d H:i:s");
  }
	
	function convertURL($text) {
		$blackList = array("Ç", "Ş", "Ğ", "Ü", "İ", "Ö", "ç", "ş", "ğ", "ü", "ö", "ı", "-");
		$whiteList = array("c", "s", "g", "u", "i", "o", "c", "s", "g", "u", "o", "i", " ");
		$link = strtolower(str_replace($blackList, $whiteList, $text));
		$link = preg_replace("@[^A-Za-z0-9\-_]@i", " ", $link);
		$link = trim(preg_replace("/\s+/", " ", $link));
		$link = str_replace(" ", "-", $link);
		return $link;
	}
	
	function convertTime($time, $type = 0, $minute = false) {
		global $translator;
		$time = strtotime($time);
		if ($type === 0) {
			$timeDifference = time() - $time;
			$second         = $timeDifference;
			$minute         = round($timeDifference/60);
			$hour           = round($timeDifference/3600);
			$day            = round($timeDifference/86400);
			$week           = round($timeDifference/604800);
			$month          = round($timeDifference/2419200);
			$year           = round($timeDifference/29030400);
			if ($second < 60) {
				if ($second === 0) {
					return t__('just now');
				}
				else {
					return t__('%s% second(s) ago', ['%s%' => $second]);
				}
			}
			else if ($minute < 60) {
				return t__('%m% minute(s) ago', ['%m%' => $minute]);
			}
			else if ($hour < 24) {
				return t__('%h% hour(s) ago', ['%h%' => $hour]);
			}
			else if ($day < 7) {
				return t__('%d% day(s) ago', ['%d%' => $day]);
			}
			else if ($week < 4) {
				return t__('%w% week(s) ago', ['%w%' => $week]);
			}
			else if ($month < 12) {
				return t__('%m% month(s) ago', ['%m%' => $month]);
			}
			else {
				return t__('%y% year(s) ago', ['%y%' => $year]);
			}
		}
		else if ($type === 1) {
			if ($minute === true) {
				return date("d.m.Y H:i", $time);
			}
			else {
				return date("d.m.Y", $time);
			}
		}
		else if ($type === 2) {
			if ($minute === true) {
				$date =  date('d.m.Y H:i', $time);
			}
			else {
				$date =  date('d.m.Y', $time);
			}
			$date   = explode('.', $date);
			$day    = $date[0];
			$month  = $date[1];
			$year   = $date[2];
			if ($month === '01') {
				$month = t__('January');
			}
			if ($month === '02') {
				$month = t__('February');
			}
			if ($month === '03') {
				$month = t__('March');
			}
			if ($month === '04') {
				$month = t__('April');
			}
			if ($month === '05') {
				$month = t__('May');
			}
			if ($month === '06') {
				$month = t__('June');
			}
			if ($month === '07') {
				$month = t__('July');
			}
			if ($month === '08') {
				$month = t__('August');
			}
			if ($month === '09') {
				$month = t__('September');
			}
			if ($month === '10') {
				$month = t__('October');
			}
			if ($month === '11') {
				$month = t__('November');
			}
			if ($month === '12') {
				$month = t__('December');
			}
			if ($minute === true) {
				$clock = explode(':', explode(' ', $year)[1]);
				$minute = $clock[0];
				$second = $clock[1];
				return sprintf("%02d %s %04d %02d:%02d", $day, $month, $year, $minute, $second);
			}
			else {
				return sprintf("%02d %s %04d", $day, $month, $year);
			}
		}
		else {
			return false;
		}
	}
	
	function convertNumber($number) {
		if ($number < 1000) {
			return number_format($number);
		}
		else if ($number < 100000) {
			return number_format($number / 1000, 1)." B";
		}
		else if ($number < 1000000) {
			return number_format($number / 1000)." B";
		}
		else if ($number < 1000000000) {
			return number_format($number / 1000000)." Mn";
		}
		else {
			return number_format($number / 1000000000)." Mr";
		}
	}

  function formatMoney($money, $removeZero = false) {
    $formattedMoney = number_format($money, 2, '.', ',');

    if ($removeZero && substr($formattedMoney, -3) == ".00")
      $formattedMoney = substr($formattedMoney, 0, -3);

    return $formattedMoney;
  }

	function money($money) {
		global $readSettings;
		$money = formatMoney($money);
		return $money.' '.$readSettings["currency"];
	}

  function credits($credits) {
    $credits = formatMoney($credits, moduleSettings('store', 'removeMoneyZeroDecimal') == 1);

    if (moduleSettings('credit', 'creditText') != "" && moduleSettings('credit', 'creditText') != null)
      return $credits." ".moduleSettings('credit', 'creditText');

    return t__('%credit% credit(s)', ['%credit%' => $credits]);
  }
	
	function hashtag($content, $character, $url) {
		$content = str_replace('&#39;', '\'', $content);
		$pattern = "/".$character."+([0-9a-zA-ZÇŞĞÜÖİçşğüöı]+)/";
		$hashtag = preg_match_all($pattern, $content, $matches);
		if ($hashtag == true) {
			for ($i=0; $i < count($matches[0]); $i++) {
				$hashtagText = $matches[0][$i];
				$hashtagURL = $url."/".convertURL($matches[1][$i]);
				$replace = '<a href="'.$hashtagURL.'">'.$hashtagText.'</a>';
				$content = str_replace($hashtagText, $replace, $content);
			}
		}
		return $content;
	}
	
	function urlContent($content) {
		$pattern = "/(http|https)\:\/\/[a-zA-Z0-9\-\.]+\.[a-zA-Z]{2,3}(\/\S*)?/";
		$url = preg_match_all($pattern, $content, $matches);
		if ($url == true) {
			for ($i=0; $i < count($matches[0]); $i++) {
				$urlText = $matches[0][$i];
				$replace = '<a href="'.$urlText.'" target="_blank" rel="nofollow">'.$urlText.'</a>';
				$content = str_replace($urlText, $replace, $content);
			}
		}
		return $content;
	}
	
	function updateSetting($name, $value) {
		global $db;
		$updateSettings = $db->prepare("UPDATE Settings SET value = ? WHERE name = ?");
		return $updateSettings->execute(array($value, $name));
	}
	
	function updateSettings($data = []) {
		global $db;
		foreach ($data as $key => $value) {
			$updateSettings = $db->prepare("UPDATE Settings SET value = ? WHERE name = ?");
			$updateSettings->execute(array($value, $key));
		}
	}

    function updateModuleSettings($module_slug, $data = []) {
        global $db;

        $module = $db->prepare("SELECT id FROM Modules WHERE slug = ?");
        $module->execute([$module_slug]);
        $module = $module->fetch();
        if (!$module) {
            abort_404();
        }

        foreach ($data as $key => $value) {
            $updateSettings = $db->prepare("UPDATE ModuleSettings SET value = ? WHERE name = ? AND moduleID = ?");
            $updateSettings->execute(array($value, $key, $module["id"]));
        }
    }
	
	function themeName() {
		global $readSettings;
		return $readSettings["themeName"];
	}
	
	function themeVersion() {
		$themeSettings = defineThemeJson();
		return isset($themeSettings["version"]) ? $themeSettings["version"] : "1.0.0";
	}
	
	function themePath($fullPath = false){
		$path = "/apps/main/app/views/themes/".themeName();
		if ($fullPath) $path = __ROOT__.$path;
		return $path;
	}
	
	function defineThemeJson() {
		static $themeSettings;
		if ($themeSettings === null) {
			$themeSettings = json_decode(file_get_contents(themePath(true)."/theme.json"), true);
		}
		
		return $themeSettings;
	}
	
	function themeJson($key) {
		$themeSettings = defineThemeJson();
		return $themeSettings[$key];
	}
	
	function getRoles($id, $fullData = false, $defaultRole = null) {
		global $db;
		
		$roles = [];
		$accountRoles = $db->prepare("SELECT R.*, AR.expiryDate FROM AccountRoles AR INNER JOIN Roles R ON AR.roleID = R.id WHERE AR.accountID = ? AND (AR.expiryDate > ? OR AR.expiryDate = ?) ORDER BY R.priority DESC");
		$accountRoles->execute(array($id, date("Y-m-d H:i:s"), '1000-01-01 00:00:00'));
		if ($accountRoles->rowCount() > 0) {
			if ($fullData)
				$roles = $accountRoles->fetchAll(PDO::FETCH_ASSOC);
			else
				foreach ($accountRoles as $readAccountRoles) {
					$roles[] = $readAccountRoles["name"];
				}
		}
		else {
			if ($defaultRole == null) {
				$defaultRole = $db->prepare("SELECT * FROM Roles WHERE slug = ?");
				$defaultRole->execute(array('default'));
				$readDefaultRole = $defaultRole->fetch(PDO::FETCH_ASSOC);
				return ($fullData) ? [$readDefaultRole] : [$readDefaultRole["name"]];
			}
			else {
				return ($fullData) ? $defaultRole : [$defaultRole["name"]];
			}
		}
		return $roles;
	}
	
	function styledRoles($roles, $defaultRole = null) {
		global $db;
		
		if (!empty($roles)) {
			return implode(", ", $roles);
		}
		else {
			if ($defaultRole == null) {
				$defaultRole = $db->prepare("SELECT * FROM Roles WHERE slug = ?");
				$defaultRole->execute(array('default'));
				$readDefaultRole = $defaultRole->fetch();
				return $readDefaultRole["name"];
			}
			else {
				return $defaultRole;
			}
		}
	}
	
	function getPermissions($id, $fullData = false) {
		$permissions = [];

		$roles = db()->prepare("SELECT * FROM AccountRoles WHERE accountID = ? AND (expiryDate > ? OR expiryDate = ?)");
		$roles->execute(array($id, date("Y-m-d H:i:s"), '1000-01-01 00:00:00'));
    $roles = $roles->fetchAll(PDO::FETCH_ASSOC);

		$rolePermissionsQuery = db()->prepare("SELECT P.id, P.name FROM RolePermissions RP INNER JOIN Permissions P ON P.id = RP.permissionID WHERE RP.roleID = ?");
		foreach ($roles as $role) {
      $rolePermissionsQuery->execute(array($role["roleID"]));
      $rolePermissions = $rolePermissionsQuery->fetchAll(PDO::FETCH_ASSOC);
			foreach ($rolePermissions as $rolePermission) {
				if ($fullData)
          $permissions[] = $rolePermission;
        else
          $permissions[] = $rolePermission["name"];
			}
		}
		$extraPermissions = db()->prepare("SELECT P.id, P.name FROM AccountPermissions AP INNER JOIN Permissions P ON P.id = AP.permissionID WHERE AP.accountID = ?");
		$extraPermissions->execute(array($id));
    $extraPermissions = $extraPermissions->fetchAll(PDO::FETCH_ASSOC);
		foreach ($extraPermissions as $extraPermission) {
      if ($fullData)
        $permissions[] = $extraPermission;
      else
        $permissions[] = $extraPermission["name"];
		}

		return $permissions;
	}
	
	function checkPerm($account, $perm) {
		if (in_array($perm, $account['permissions']) || in_array('SUPER_ADMIN', $account['permissions'])) {
			return true;
		}
		else {
			return false;
		}
	}
	
	function checkStaff($account) {
		return checkPerm($account, 'VIEW_DASHBOARD');
	}
	
	function getSupportPermissions($id) {
		$permissions = [];
		
		$permissions['STRICT_MANAGE_SUPPORT_TICKETS'] = false;
		
		$roles = db()->prepare("SELECT * FROM AccountRoles WHERE accountID = ? AND (expiryDate > ? OR expiryDate = ?)");
		$roles->execute(array($id, date("Y-m-d H:i:s"), '1000-01-01 00:00:00'));
		$roles = $roles->fetchAll(PDO::FETCH_ASSOC);
		
		$rolePermissionsQuery = db()->prepare("SELECT P.id, P.name, RP.categoryID FROM SupportCategoryRolePermissions RP INNER JOIN SupportCategoryPermissions P ON P.id = RP.permissionID WHERE RP.roleID = ?");
		foreach ($roles as $role) {
			$rolePermissionsQuery->execute(array($role["roleID"]));
			$rolePermissions = $rolePermissionsQuery->fetchAll(PDO::FETCH_ASSOC);
			foreach ($rolePermissions as $rolePermission) {
				$permissions[$rolePermission["name"]][] = $rolePermission["categoryID"];
			}
		}
		$extraPermissions = db()->prepare("SELECT P.id, P.name FROM AccountPermissions AP INNER JOIN Permissions P ON P.id = AP.permissionID WHERE AP.accountID = ? AND P.name = ? LIMIT 1");
		$extraPermissions->execute(array($id, 'MANAGE_SUPPORT_TICKETS'));
		$extraPermissions = $extraPermissions->fetch(PDO::FETCH_ASSOC);
		if ($extraPermissions) {
			$permissions['STRICT_MANAGE_SUPPORT_TICKETS'] = true;
		}
		
		return $permissions;
	}
	
	function styledUsername($username, $role, $link = true, $extraClass = null) {
		if ($link)
			return '<a href="/player/'.$username.'" class="username__'.$role.' '.$extraClass.'">' . $username . '</a>';
		else
			return '<span class="username__'.$role.' '.$extraClass.'">' . $username . '</span>';
	}
	
	function checkPassword($type, $password, $hash) {
		if ($type == 1)
			return checkSHA256($password, $hash);
		elseif ($type == 2)
			return md5($password) == $hash;
		elseif ($type == 3)
			return password_verify($password, $hash);
		else
			return false;
	}
	
	function createPassword($type, $password) {
		if ($type == 1)
			return createSHA256($password);
		elseif ($type == 2)
			return md5($password);
		elseif ($type == 3)
			return password_hash($password, PASSWORD_BCRYPT);
		else
			return null;
	}
	
	function createLog($action, $data = null, $user = null) {
		global $db;

    if ($user == null)
      $user = auth()->user()->id();

    if ($data != null && is_array($data))
      $data = json_encode($data);
		
		$insertLogs = $db->prepare("INSERT INTO Logs (accountID, action, data, ip, creationDate) VALUES (?, ?, ?, ?, ?)");
		$insertLogs->execute(array($user, $action, $data, getIP(), date("Y-m-d H:i:s")));
	}
	
	function moduleIsDisabled($module) {
		global $readModules;
		
		foreach ($readModules as $readModule) {
			if ($readModule["slug"] == $module) {
				return $readModule["isEnabled"] == 0;
			}
		}
		return false;
	}

function checkModuleStatus($modules, $strict = false): bool
{
  global $readModules;

  if (!is_array($modules)) {
    foreach ($readModules as $readModule) {
      if ($readModule["slug"] === $modules && $readModule["isEnabled"] == 1) {
        return true;
      }
    }
    return false;
  }

  foreach ($modules as $module) {
    $moduleStatus = checkModuleStatus($module);

    if ($strict && !$moduleStatus) {
      return false;
    } elseif (!$strict && $moduleStatus) {
      return true;
    }
  }

  return $strict;
}

function websiteUrl($path = null)
{
	$baseURL = settings('websiteURL');
	if (settings('websiteURL') == '') {
		$baseURL = ((!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off' || $_SERVER['SERVER_PORT'] == 443) ? "https://" : "http://") . $_SERVER['HTTP_HOST'];
	}
	return rtrim($baseURL, '/') . ($path ? "/" . trim($path, '/') : "");
}

function getCategoryListRecursive($categoryID, $list = [])
{
  $category = db()->prepare("SELECT id, name, parentID FROM ProductCategories WHERE id = ?");
  $category->execute(array($categoryID));
  $category = $category->fetch(PDO::FETCH_ASSOC);

  if (!$category) return [];

  $list[$category["id"]] = $category["name"];

  if ($category["parentID"] != 0) {
    return getCategoryListRecursive($category["parentID"], $list);
  }

  return $list;
}

function escape_t__($string, $vars = [])
{
  return str_replace("'", "\'", t__($string, $vars));
}

function escape_e__($string, $vars = [])
{
  echo escape_t__($string, $vars);
}

function url_is($url): bool
{
	return request()->getUrl() == url($url);
}