<?php

namespace Main\Controllers\Account;

use Main\Core\Controller;
use Main\Core\Jobs\JobDispatcher;
use Main\Core\Redirect;
use Main\Jobs\SendEmail;
use Main\Services\ImageCropService;
use Main\Services\PasswordService;
use Main\Services\SeoService;
use Main\Services\AccountLinkingService;
use PDO;
use PHPMailer\PHPMailer\Exception;

class SettingsController extends Controller
{
		protected string $route_type = 'account-settings';
	
		public function profile()
		{
			$this->route_type = 'account-profile-settings';
			
			SeoService::set('account.settings.profile');
			
			$customFields = db()->prepare("SELECT ACF.id, ACF.name, ACF.description, ACF.isRequired, ACFV.value FROM AccountCustomFields ACF LEFT JOIN AccountCustomFieldValues ACFV ON ACF.id = ACFV.customFieldID AND ACFV.accountID = ?");
			$customFields->execute(array(auth()->user()->id()));
			$customFields = $customFields->fetchAll(PDO::FETCH_ASSOC);
			
			$contactInfo = db()->prepare("SELECT * FROM AccountContactInfo WHERE accountID = ?");
			$contactInfo->execute(array(auth()->user()->id()));
			$contactInfo = $contactInfo->fetch();
			if (!$contactInfo) {
				$contactInfo = [
					"firstName" => "",
					"lastName" => "",
					"phoneNumber" => ""
				];
			}
			
			return $this->view('account.settings.edit-profile', compact('customFields', 'contactInfo'));
		}
  
		public function security()
    {
	      $this->route_type = 'account-security-settings';
			
	      SeoService::set('account.settings.security');
	    
		    $tfaStatus = false;
		    if (modules('tfa')->isActive()) {
			    $tfaStatus = db()->prepare("SELECT * FROM AccountTfaKeys WHERE accountID = ?");
			    $tfaStatus->execute(array(auth()->user()->id()));
			    $tfaStatus = $tfaStatus->rowCount() > 0;
		    }
			
        return $this->view('account.settings.security', compact('tfaStatus'));
    }
	
		public function linkedAccounts()
		{
			if (!modules('account_linking')->isActive()) {
				abort_404();
			}
			
			$this->route_type = 'account-linked-accounts';
			
			SeoService::set('account.settings.linked-accounts');
			
			$providers = AccountLinkingService::getProviders();
			$providers = array_map(function ($provider) {
				$provider["isLinked"] = false;
				$linkedAccount = db()->prepare("SELECT * FROM LinkedAccounts WHERE accountID = ? AND providerID = ?");
				$linkedAccount->execute([auth()->user()->id(), $provider["id"]]);
				$linkedAccount = $linkedAccount->fetch();
				if ($linkedAccount) {
					$provider["isLinked"] = true;
					$provider["connection"] = $linkedAccount;
				}
				
				return $provider;
			}, $providers);
			
			return $this->view('account.settings.linked-accounts', compact('providers'));
		}
		
		public function uploadAvatar()
		{
			if (settings('allowCustomAvatars') == 0) abort_404();
			
			validate([
				'avatar' => 'required|uploaded_file|max:2M|mimes:jpeg,jpg,png,gif'
			]);
			
			$avatar = $_FILES["avatar"];
			
			$imageID = rand_token();
			$image = new \Bulletproof\Image($avatar);
			$image->setStorage(__ROOT__ . "/apps/main/public/images/avatars/");
			$image->setMime(array('png', 'jpg', 'jpeg', 'gif'));
			$image->setSize(10, 2 * 1000000); // 2 MB
			$image->setName($imageID);
			
			$upload = $image->upload();
			if ($upload) {
				try {
					// Crop and Resize Avatar
					ImageCropService::processAvatar($upload->getPath(), 256);
				} catch (\Exception $e) {
					return back()->flash("error", t__('An error occupied while uploading an image: %error%', ['%error%' => $e->getMessage()]));
				}
				
				$imageType = $image->getMime();
				
				$currentAvatar = db()->prepare("SELECT * FROM Avatars WHERE accountID = ?");
				$currentAvatar->execute(array(auth()->user()->id()));
				$currentAvatar = $currentAvatar->fetch();
				if ($currentAvatar) {
					$updateAvatar = db()->prepare("UPDATE Avatars SET imageID = ?, imageType = ? WHERE accountID = ?");
					$updateAvatar->execute(array($imageID, $imageType, auth()->user()->id()));
					
					// Delete the old avatar
					@unlink(__ROOT__ . "/apps/main/public/images/avatars/" . $currentAvatar["imageID"] . "." . $currentAvatar["imageType"]);
				} else {
					$insertAvatar = db()->prepare("INSERT INTO Avatars (accountID, imageID, imageType) VALUES (?, ?, ?)");
					$insertAvatar->execute(array(auth()->user()->id(), $imageID, $imageType));
				}
				
				return back()->flash("success", t__('Your avatar has been successfully updated!'));
			}
			else {
				return back()->flash("error", t__('An error occupied while uploading an image: %error%', ['%error%' => $image->getError()]));
			}
		}
	
	public function removeAvatar()
	{
		$avatar = db()->prepare("SELECT * FROM Avatars WHERE accountID = ?");
		$avatar->execute(array(auth()->user()->id()));
		$avatar = $avatar->fetch();
		if (!$avatar) return back()->flash("error", t__('Avatar not found!'));
		
		$deleteAvatar = db()->prepare("DELETE FROM Avatars WHERE accountID = ?");
		$deleteAvatar->execute(array(auth()->user()->id()));
		
		// Delete Image
		@unlink(__ROOT__ . "/apps/main/public/images/avatars/" . $avatar["imageID"] . "." . $avatar["imageType"]);
		
		return back()->flash("success", t__('Avatar removed successfully.'));
	}
	
		public function updateDetails()
		{
			validate([
				'firstName' => 'required',
				'lastName' => 'required',
				'phoneNumber' => 'required'
			]);
			
			$checkAccountContactInfo = db()->prepare("SELECT * FROM AccountContactInfo WHERE accountID = ?");
			$checkAccountContactInfo->execute(array(auth()->user()->id()));
			$checkAccountContactInfo = $checkAccountContactInfo->fetch();
			
			$firstName = sanitize(input("firstName"));
			$lastName = sanitize(input("lastName"));
			$phoneNumber = sanitize(input("phoneNumber"));
			
			if ($checkAccountContactInfo) {
				$updateAccountContactInfo = db()->prepare("UPDATE AccountContactInfo SET firstName = ?, lastName = ?, phoneNumber = ? WHERE accountID = ?");
				$updateAccountContactInfo->execute(array($firstName, $lastName, $phoneNumber, auth()->user()->id()));
			} else {
				$insertAccountContactInfo = db()->prepare("INSERT INTO AccountContactInfo (accountID, firstName, lastName, phoneNumber) VALUES (?, ?, ?, ?)");
				$insertAccountContactInfo->execute(array(auth()->user()->id(), $firstName, $lastName, $phoneNumber));
			}
			
			foreach ($_POST as $fieldKey => $fieldValue) {
				if (substr($fieldKey, 0, 13) == 'customfield__') {
					$fieldID = substr($fieldKey, 13);
					$customFieldValue = db()->prepare("SELECT * FROM AccountCustomFieldValues WHERE accountID = ? AND customFieldID = ?");
					$customFieldValue->execute(array(auth()->user()->id(), $fieldID));
					$customFieldValue = $customFieldValue->fetchAll(PDO::FETCH_ASSOC);
					$input = sanitize(input($fieldKey));
					if (count($customFieldValue) > 0) {
						$insertCustomFieldValue = db()->prepare("UPDATE AccountCustomFieldValues SET value = ? WHERE accountID = ? AND customFieldID = ?");
						$insertCustomFieldValue->execute(array($input, auth()->user()->id(), $fieldID));
					} else {
						$insertCustomFieldValue = db()->prepare("INSERT INTO AccountCustomFieldValues (accountID, customFieldID, value) VALUES (?, ?, ?)");
						$insertCustomFieldValue->execute(array(auth()->user()->id(), $fieldID, $input));
					}
				}
			}
			
			return back()->flash("success", t__('Your profile has been updated.'));
		}
		
		public function changeEmail()
		{
			validate([
				'email' => 'required|email',
			]);
			
			$newEmail = sanitize(input("email"));
			
			$emailValid = db()->prepare("SELECT * FROM Accounts WHERE email = ?");
			$emailValid->execute(array($newEmail));
			
			if (!is_email($newEmail)) {
				return back()->flash("error", t__('Please enter a valid email!'));
			}
			if ($newEmail != auth()->user()->email() && $emailValid->rowCount() > 0) {
				return back()->flash("error", t__('This email is already in use!'));
			}
			
			if (auth()->user()->email() != $newEmail) {
				if (settings('emailChangeVerification') == 1 && auth()->user()->isEmailVerified() && auth()->user()->email() != "your@email.com") {
					try {
						$changeRequest = db()->prepare("SELECT * FROM AccountEmailChangeTokens WHERE accountID = ? AND newEmail = ? AND expiryDate > ?");
						$changeRequest->execute(array(auth()->user()->id(), $newEmail, datetime()));
						$changeRequest = $changeRequest->fetch();
						
						if ($changeRequest) {
							// Last try is less than 2 min ago
							if ($changeRequest["lastSend"] > date('Y-m-d H:i:s', strtotime('-2 minutes'))) {
								return back()->flash('error', t__('Please wait a few minutes before trying again.'));
							}
							$token = $changeRequest['token'];
							$updateAccountPasswordResetTokens = db()->prepare("UPDATE AccountEmailChangeTokens SET lastSend = ? WHERE id = ?");
							$updateAccountPasswordResetTokens->execute(array(datetime(), $changeRequest['id']));
						} else {
							$token = rand_token();
							$insertAccountPasswordResetTokens = db()->prepare("INSERT INTO AccountEmailChangeTokens (accountID, token, oldEmail, newEmail, creationIP, expiryDate, lastSend, creationDate) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
							$insertAccountPasswordResetTokens->execute(array(auth()->user()->id(), $token, auth()->user()->email(), $newEmail, getIP(), createDuration(0.04166666666), datetime(), datetime()));
						}
						
						$url = website_url(url('account.settings.security.confirm-email-change', ['token' => $token]));
						$search = array("%username%", "%email%", "%url%");
						$replace = array(auth()->user()->displayName(), $newEmail, $url);
						$template = settings("smtpVerifyEmailChangeTemplate");
						$content = str_replace($search, $replace, $template);
						JobDispatcher::dispatch((new SendEmail(auth()->user()->email(), settings('smtpVerifyEmailChangeTitle'), $content)));
						
						return back()->flash("warning", t__('A confirmation email has been sent to <strong>%email%</strong>. Please click the link in that email to confirm the email address change.', ['%email%' => auth()->user()->email()]));
					} catch (Exception $e) {
						return back()->flash('error', t__('Could not send mail due to a system error:')." ".$e->errorMessage());
					}
				}
				
				$loginToken = md5(uniqid(mt_rand(), true));
				
				$updateAccounts = db()->prepare("UPDATE Accounts SET email = ? WHERE id = ?");
				$updateAccounts->execute(array($newEmail, auth()->user()->id()));
				
				$deleteAccountSessions = db()->prepare("DELETE FROM AccountSessions WHERE accountID = ?");
				$deleteAccountSessions->execute(array(auth()->user()->id()));
				
				$insertAccountSessions = db()->prepare("INSERT INTO AccountSessions (accountID, loginToken, useragent, creationIP, expiryDate, creationDate) VALUES (?, ?, ?, ?, ?, ?)");
				$insertAccountSessions->execute(array(auth()->user()->id(), $loginToken, $_SERVER['HTTP_USER_AGENT'], getIP(), createDuration(((cookies()->has('rememberMe')) ? 365 : 0.01666666666)), datetime()));
				
				session()->set("login", $loginToken);
				if (cookies()->has('rememberMe')) {
					cookies()->set('rememberMe', $loginToken, 365 * 60 * 60 * 24);
				}
				
				return back()->flash('success', t__('Your email has been changed!'));
			}
			
			return back();
		}
	
		public function confirmEmailChange($token)
		{
			if (settings('emailChangeVerification') != 1) abort_404();
			
			$changeRequest = db()->prepare("SELECT * FROM AccountEmailChangeTokens WHERE accountID = ? AND token = ? AND expiryDate > ?");
			$changeRequest->execute(array(auth()->user()->id(), $token, datetime()));
			$changeRequest = $changeRequest->fetch();
			
			if (!$changeRequest) abort_404();
			
			$expireTokens = db()->prepare("UPDATE AccountEmailChangeTokens SET expiryDate = ? WHERE accountID = ?");
			$expireTokens->execute(array(datetime(), auth()->user()->id()));
			
			$loginToken = md5(uniqid(mt_rand(), true));
			
			$updateAccounts = db()->prepare("UPDATE Accounts SET email = ? WHERE id = ?");
			$updateAccounts->execute(array($changeRequest["newEmail"], auth()->user()->id()));
			
			$deleteAccountSessions = db()->prepare("DELETE FROM AccountSessions WHERE accountID = ?");
			$deleteAccountSessions->execute(array(auth()->user()->id()));
			
			$insertAccountSessions = db()->prepare("INSERT INTO AccountSessions (accountID, loginToken, useragent, creationIP, expiryDate, creationDate) VALUES (?, ?, ?, ?, ?, ?)");
			$insertAccountSessions->execute(array(auth()->user()->id(), $loginToken, $_SERVER['HTTP_USER_AGENT'], getIP(), createDuration(((cookies()->has('rememberMe')) ? 365 : 0.01666666666)), datetime()));
			
			session()->set("login", $loginToken);
			if (cookies()->has('rememberMe')) {
				cookies()->set('rememberMe', $loginToken, 365 * 60 * 60 * 24);
			}
			
			$redirect = new Redirect(url('account.settings.security'));
			return $redirect->flash('success', t__('Your email has been changed!'));
		}

    public function changePassword()
    {
        validate([
            "currentPassword" => "required",
            "password" => "required|min:4",
            "passwordRe" => "required|same:password"
        ]);
				
				if (settings('loginProvider') != 'default') abort_404();
	    
		    $userPassword = db()->prepare("SELECT password FROM Accounts WHERE id = ?");
		    $userPassword->execute([auth()->user()->id()]);
		    $userPassword = $userPassword->fetchColumn();
		    $currentPassword = PasswordService::match(input("currentPassword"), $userPassword);
				
        if (!$currentPassword) {
            return back()->flash("error", t__('Your current password is incorrect!'));
        }
        if (!validate_password(input("password"))) {
            return back()->flash("error", t__('Your password is too weak!'));
        }

        $loginToken = rand_token();
		    $password = PasswordService::hash(input("password"));

        $updateAccounts = db()->prepare("UPDATE Accounts SET password = ? WHERE id = ?");
        $updateAccounts->execute(array($password, auth()->user()->id()));
				
        $deleteAccountSessions = db()->prepare("DELETE FROM AccountSessions WHERE accountID = ?");
        $deleteAccountSessions->execute(array(auth()->user()->id()));
        
				$insertAccountSessions = db()->prepare("INSERT INTO AccountSessions (accountID, loginToken, useragent, creationIP, expiryDate, creationDate) VALUES (?, ?, ?, ?, ?, ?)");
        $insertAccountSessions->execute(array(auth()->user()->id(), $loginToken, $_SERVER['HTTP_USER_AGENT'], getIP(), createDuration(((cookies()->has('rememberMe')) ? 365 : 0.01666666666)), datetime()));
				
				session()->set("login", $loginToken);
		    if (cookies()->has('rememberMe')) {
			    cookies()->set('rememberMe', $loginToken, 365 * 60 * 60 * 24);
		    }

        return back()->flash("success", t__('Your password has been changed!'));
    }
	
		public function toggleTfa()
		{
			$tfaKeys = db()->prepare("SELECT * FROM AccountTfaKeys WHERE accountID = ?");
			$tfaKeys->execute([auth()->user()->id()]);
			$isActive = $tfaKeys->rowCount() > 0;
			
			// If the user has enabled TFA and the user wants to disable it, delete the TFA keys
			if ($isActive) {
				$deleteTfaKeys = db()->prepare("DELETE FROM AccountTfaKeys WHERE accountID = ?");
				$deleteTfaKeys->execute(array(auth()->user()->id()));
				session()->remove('isTfaRequired');
				
				return back()->flash("success", t__('Two Factor Authentication has been disabled.'));
			}
			
			// If the user has not enabled TFA and the user wants to enable it, redirect to the TFA setup page
			redirect(url("tfa.setup.index"));
		}
}