<?php

namespace Main\Controllers\Store;

use Exception;
use Main\Core\Controller;
use Main\Core\Jobs\JobDispatcher;
use Main\Jobs\SendDiscordWebhook;
use Main\Jobs\SendEmail;
use Main\Libs\Payment\Order;
use Main\Libs\Payment\Product;
use Main\Services\CartService;
use Main\Services\DeliveryService;
use Main\Services\EmailService;
use Main\Services\PaymentGatewayService;
use Main\Services\SeoService;
use PDO;
use Rakit\Validation\Validator;

class CartController extends Controller
{
		protected string $route_type = 'cart';
	
		public function __construct()
		{
			if (modules('store')->settings('redirect') != '') {
				redirect(modules('store')->settings('redirect'));
			}
		}

    public function index()
    {
	      SeoService::set('store.cart');
				
	      $paymentGateways = db()->prepare("SELECT * FROM PaymentGateways WHERE status = ? ORDER BY `order` ASC, title ASC");
				$paymentGateways->execute(array(1));
				$paymentGateways = $paymentGateways->fetchAll();
				
				$creatorCodesCount = db()->query("SELECT COUNT(*) FROM CreatorCodes")->fetchColumn();
				$showCreatorCodesInput = $creatorCodesCount > 0;

        return $this->view('store.cart', compact('paymentGateways', 'showCreatorCodesInput'));
    }
		
		public function items()
		{
			response()->json(CartService::getCart());
		}

    public function addItem()
    {
				// If the request method is GET
	      if (request()->getMethod() == 'get') {
					if (!isset($_GET["product"])) {
						redirect(url('store.index'));
					}
					
					$productID = $_GET["product"];
					$quantity = $_GET["quantity"] != null ? $_GET["quantity"] : 1;
		      
		      $product = db()->prepare("SELECT P.*, C.isCumulative FROM Products P INNER JOIN ProductCategories C ON P.categoryID = C.id WHERE P.id = ? AND P.isActive = ?");
		      $product->execute(array($productID, 1));
		      $product = $product->fetch();
		      
		      if (!$product) {
			      return back()->flash('error', t__("Something went wrong! Please try again later."));
		      }
					
					if ($product["disableQuantity"] == '1' && $quantity > 1) {
						$quantity = 1;
					}
		      
		      $variables = db()->prepare("SELECT V.* FROM ProductVariables PV INNER JOIN StoreVariables V ON PV.variableID = V.id WHERE PV.productID = ?");
		      $variables->execute(array($product["id"]));
		      $product["variables"] = array_map(function ($variable) {
			      $variable["options"] = [];
						if ($variable["type"] == "dropdown") {
							$dropdownOptions = db()->prepare("SELECT * FROM StoreVariableDropdownOptions WHERE variableID = ?");
							$dropdownOptions->execute(array($variable["id"]));
							$variable["options"] = $dropdownOptions->fetchAll(PDO::FETCH_ASSOC);
						}
						
						return $variable;
		      }, $variables->fetchAll(PDO::FETCH_ASSOC));
					
					if (count($product["variables"]) == 0) {
						redirect(url('store.products.show', ['id' => $product["id"]]));
					}
					
					return $this->view('store.add-cart', compact('product', 'quantity'));
	      }
	    
	      // If the request method is POST
	    
		    $validate = validate([
			    'productID' => 'required|numeric',
			    'quantity' => 'default:1|required|numeric|min:1',
		    ], false);
		    
		    $isJson = request()->isAjax() || request()->isFormatAccepted('json');
		    
		    if (count($validate) > 0) {
			    if ($isJson) {
				    response()->json([
					    'status' => false,
					    'errors' => $validate,
				    ]);
			    }
			    return back()->flash('error', $validate);
		    }
		    
		    $productID = input("productID");
		    $quantity = input("quantity");
				
		    // Create a shopping cart if not exists
		    $shoppingCart = db()->prepare("SELECT * FROM ShoppingCarts WHERE accountID = ?");
		    $shoppingCart->execute(array(auth()->user()->id()));
		    $shoppingCart = $shoppingCart->fetch();
		    if (!$shoppingCart) {
			    $createShoppingCart = db()->prepare("INSERT INTO ShoppingCarts (accountID) VALUES (?)");
			    $createShoppingCart->execute(array(auth()->user()->id()));
		    }
				
				$shoppingCartID = auth()->user()->id();
				
        $product = db()->prepare("SELECT P.*, C.isCumulative FROM Products P INNER JOIN ProductCategories C ON P.categoryID = C.id WHERE P.id = ? AND P.isActive = ?");
        $product->execute(array($productID, 1));
        $product = $product->fetch();
				
				if (!$product) {
					if ($isJson) {
						response()->json([
							'status' => false,
							'error' => 'error_product',
							'message' => t__("Something went wrong! Please try again later."),
						]);
					}
					return back()->flash('error', t__("Something went wrong! Please try again later."));
				}
	    
		    // Check stock out
		    if ($product["stock"] == 0) {
			    if ($isJson) {
				    response()->json([
					    'status' => false,
					    'error' => 'error_stock',
					    'message' => t__("Not enough items in the stock!"),
				    ]);
			    }
			    return back()->flash('error', t__("Not enough items in the stock!"));
		    }
				
				// Block downgrading
		    if ($product["isCumulative"] == '1') {
			    $subs = db()->prepare("SELECT * FROM Subscriptions S INNER JOIN Products P ON P.id = S.productID WHERE P.categoryID = ? AND accountID = ? AND (endsAt > ? OR endsAt = '1000-01-01 00:00:00') ORDER BY P.priority ASC LIMIT 1");
			    $subs->execute(array($product["categoryID"], auth()->user()->id(), datetime()));
			    $subs = $subs->fetch(PDO::FETCH_ASSOC);
			    if ($subs && $product["priority"] >= $subs["priority"]) {
				    if ($isJson) {
					    response()->json([
						    'status' => false,
						    'error' => 'error_downgrade',
						    'message' => t__("You can't downgrade your subscription!"),
					    ]);
				    }
				    return back()->flash('error', t__("You can't downgrade your subscription!"));
			    }
				}
				
				// If the product is cumulative, check if there is another product in the cart with the same category. If there is, create an array to delete after adding the new product
		    $otherProductsInCartWithSameCumulativeCategory = [];
				if ($product["isCumulative"] == '1') {
					$otherProductsInCartWithSameCumulativeCategoryQuery = db()->prepare("SELECT P.id FROM ShoppingCartProducts SCP INNER JOIN Products P ON SCP.productID = P.id INNER JOIN ProductCategories C ON P.categoryID = C.id WHERE SCP.shoppingCartID = ? AND C.id = ? AND P.id != ?");
					$otherProductsInCartWithSameCumulativeCategoryQuery->execute(array($shoppingCartID, $product["categoryID"], $product["id"]));
					$otherProductsInCartWithSameCumulativeCategory = $otherProductsInCartWithSameCumulativeCategoryQuery->fetchAll(PDO::FETCH_COLUMN);
				}
	   
				// Get the cart items with the same product
		    $cartItems = db()->prepare("SELECT * FROM ShoppingCartProducts WHERE shoppingCartID = ? AND productID = ?");
		    $cartItems->execute(array($shoppingCartID, $product["id"]));
		    $cartItems = $cartItems->fetchAll(PDO::FETCH_ASSOC);
	    
		    // Is disabled quantity
	      $isDisabledQuantity = $product["isCumulative"] == '1';
		    if ($isDisabledQuantity) {
			    if ($quantity > 1 || count($cartItems) > 0) {
				    if ($isJson) {
					    response()->json([
						    'status' => false,
						    'error' => 'error_disabled_quantity',
						    'message' => t__("You can only purchase one of this product!"),
					    ]);
				    }
						redirect(url('cart.index'));
			    }
		    }
				
				$variables = db()->prepare("SELECT V.* FROM ProductVariables PV INNER JOIN StoreVariables V ON PV.variableID = V.id WHERE PV.productID = ?");
				$variables->execute(array($product["id"]));
				$product["variables"] = $variables->fetchAll(PDO::FETCH_ASSOC);
				if (count($product["variables"]) > 0) {
					foreach ($product["variables"] as $variable) {
						// If any required variable is not provided redirect to add cart page to fill the variables
						if (!isset($_POST["variable_" . $variable["id"]]) || $_POST["variable_" . $variable["id"]] === null) {
							if ($isJson) {
								response()->json([
									'status' => false,
									'redirect' => url('cart.items.add', null, ['product' => $product["id"], 'quantity' => $quantity]),
								]);
							}
							
							redirect(url('cart.items.add', null, ['product' => $product["id"], 'quantity' => $quantity]));
						}
						else {
							// Validate variables
							$rule = "required";
							if ($variable["type"] == "number") {
								$rule = "required|numeric";
							}
							if ($variable["type"] == "email") {
								$rule = "required|email";
							}
							if ($variable["type"] == "url") {
								$rule = "required|url";
							}
							if ($variable["type"] == "dropdown") {
								$dropdownOptions = db()->prepare("SELECT * FROM StoreVariableDropdownOptions WHERE variableID = ?");
								$dropdownOptions->execute(array($variable["id"]));
								$dropdownOptions = $dropdownOptions->fetchAll(PDO::FETCH_ASSOC);
								$validator = new Validator;
								$rule = [
									"required",
									$validator('in', array_column($dropdownOptions, 'value'))
								];
							}
							$validateVariables = validate([
								'variable_' . $variable["id"] => $rule,
							], false);
							
							if (count($validateVariables) > 0) {
								if ($isJson) {
									response()->json([
										'status' => false,
										'errors' => $validateVariables,
									]);
								}
								return back()->flash('error', $validateVariables);
							}
						}
					}
				}
				
				// Check required linked accounts
        if (modules('account_linking')->isActive()) {
	        $requiredLinkedAccounts = db()->prepare("SELECT requiredProviderID FROM ProductRequiredLinkedAccounts WHERE productID = ?");
	        $requiredLinkedAccounts->execute(array($product["id"]));
	        $requiredLinkedAccounts = $requiredLinkedAccounts->fetchAll(PDO::FETCH_ASSOC);
	        foreach ($requiredLinkedAccounts as $requiredLinkedAccount) {
						$accountLinkingProvider = db()->prepare("SELECT name FROM AccountLinkingProviders WHERE id = ? AND isEnabled = ?");
						$accountLinkingProvider->execute(array($requiredLinkedAccount["requiredProviderID"], 1));
						$accountLinkingProvider = $accountLinkingProvider->fetch(PDO::FETCH_ASSOC);
						if (!$accountLinkingProvider) continue;
						
		        $linkedAccount = db()->prepare("SELECT identifier FROM LinkedAccounts WHERE accountID = ? AND providerID = ?");
		        $linkedAccount->execute(array(auth()->user()->id(), $requiredLinkedAccount["requiredProviderID"]));
		        $linkedAccount = $linkedAccount->fetch(PDO::FETCH_ASSOC);
		        if (!$linkedAccount) {
			        if ($isJson) {
				        response()->json([
					        'status' => false,
					        'error' => 'error_linked_account',
					        'message' => t__("You must link your %provider% account before purchasing this product!", ['%provider%' => t__($accountLinkingProvider["name"])]),
				        ]);
			        }
			        return back()->flash('error', t__("You must link your %provider% account before purchasing this product!", ['%provider%' => t__($accountLinkingProvider["name"])]));
		        }
	        }
        }
	    
				// Check required products
		    $purchasedProductsIDs = db()->prepare("SELECT productID FROM PurchasedProducts WHERE accountID = ? AND (expiresAt > ? OR expiresAt IS NULL)");
		    $purchasedProductsIDs->execute(array(auth()->user()->id(), datetime()));
		    $purchasedProductsIDs = $purchasedProductsIDs->fetchAll(PDO::FETCH_COLUMN);
		    $requiredProductIDs = db()->prepare("SELECT requiredProductID FROM ProductRequiredProducts WHERE productID = ?");
		    $requiredProductIDs->execute(array($product["id"]));
		    $requiredProductIDs = $requiredProductIDs->fetchAll(PDO::FETCH_COLUMN);
		    
		    if ($requiredProductIDs) {
			    if (!$purchasedProductsIDs) {
				    if ($isJson) {
					    response()->json([
						    'status' => false,
						    'error' => 'error_required_product',
						    'message' => t__("You must purchase the required product before purchasing this product!"),
					    ]);
				    }
				    return back()->flash('error', t__("You must purchase the required product before purchasing this product!"));
			    }
			    if ($product["requireOnlyOneProduct"] == '1') {
				    if (empty(array_intersect($requiredProductIDs, $purchasedProductsIDs))) {
					    if ($isJson) {
						    response()->json([
							    'status' => false,
							    'error' => 'error_required_product',
							    'message' => t__("You must purchase the required product before purchasing this product!"),
						    ]);
					    }
					    return back()->flash('error', t__("You must purchase the required product before purchasing this product!"));
				    }
			    } else {
				    $intersection = array_intersect($purchasedProductsIDs, $requiredProductIDs);
				    if (count($intersection) != count($requiredProductIDs)) {
					    if ($isJson) {
						    response()->json([
							    'status' => false,
							    'error' => 'error_required_product',
							    'message' => t__("You must purchase the required product before purchasing this product!"),
						    ]);
					    }
					    return back()->flash('error', t__("You must purchase the required product before purchasing this product!"));
				    }
			    }
		    }
		    
		    $categoryRequiredProductIDs = db()->prepare("SELECT productID FROM ProductCategoryRequiredProducts WHERE categoryID = ?");
		    $categoryRequiredProductIDs->execute(array($product["categoryID"]));
		    $categoryRequiredProductIDs = $categoryRequiredProductIDs->fetchAll(PDO::FETCH_COLUMN);
		    if ($categoryRequiredProductIDs && empty(array_intersect($purchasedProductsIDs, $categoryRequiredProductIDs))) {
			    if ($isJson) {
				    response()->json([
					    'status' => false,
					    'error' => 'error_required_product',
					    'message' => t__("You must purchase the required product before purchasing this product!"),
				    ]);
			    }
			    return back()->flash('error', t__("You must purchase the required product before purchasing this product!"));
		    }
	    
	      // Total quantity of the product in the cart, we will use it for stock check
				$totalQuantity = 0;
				
				// Try to find same product with same variables in the cart
				$cartItem = null;
				foreach ($cartItems as $cartItemElement) {
					// Add the cart item quantity to the total quantity
					$totalQuantity += $cartItemElement["quantity"];
					
					// Get the product variables
					$shoppingCartProductVariableValues = db()->prepare("SELECT * FROM ShoppingCartProductVariableValues WHERE cartProductID = ?");
					$shoppingCartProductVariableValues->execute(array($cartItemElement["id"]));
					$shoppingCartProductVariableValues = $shoppingCartProductVariableValues->fetchAll(PDO::FETCH_ASSOC);
					
					// Create an array for provided product variables, we will use it for comparison
					$providedProductVariables = [];
					foreach ($product["variables"] as $variable) {
						$providedProductVariables[] = [
							"id" => $variable["id"],
							"value" => sanitize(input('variable_' . $variable["id"]))
						];
					}
					usort($providedProductVariables, fn($a, $b) => $a['id'] <=> $b['id']);
					
					// Create an array for cart item variables, we will use it for comparison
					$cartItemVariables = [];
					foreach ($shoppingCartProductVariableValues as $variable) {
						$cartItemVariables[] = [
							"id" => $variable["variableID"],
							"value" => $variable["value"]
						];
					}
					usort($cartItemVariables, fn($a, $b) => $a['id'] <=> $b['id']);
					
					// If we found the same product with the same variables in the cart
					if (json_encode($cartItemVariables) == json_encode($providedProductVariables)) {
						$cartItem = $cartItemElement;
						break;
					}
				}
	   
				// Add the quantity of items added from the post request to the total quantity
		    $totalQuantity += $quantity;
				
				if ($product["stock"] != -1 && $totalQuantity > $product["stock"]) {
					if ($isJson) {
						response()->json([
							'status' => false,
							'error' => 'error_stock',
							'message' => t__("Not enough items in the stock!"),
						]);
					}
					return back()->flash('error', t__("Not enough items in the stock!"));
				}
	   
				// If the product is cumulative, delete the other products in the cart with the same category
		    if (count($otherProductsInCartWithSameCumulativeCategory) > 0) {
			    $deleteShoppingCartItem = db()->prepare("DELETE FROM ShoppingCartProducts WHERE shoppingCartID = ? AND productID = ?");
			    foreach ($otherProductsInCartWithSameCumulativeCategory as $otherProduct) {
				    $deleteShoppingCartItem->execute(array($shoppingCartID, $otherProduct["id"]));
			    }
		    }
	   
				// If the product not added to the cart add it
		    if ($cartItem === null) {
			    $insertShoppingCartItem = db()->prepare("INSERT INTO ShoppingCartProducts (shoppingCartID, productID, quantity) VALUES (?, ?, ?)");
			    $insertShoppingCartItem->execute(array($shoppingCartID, $product["id"], $quantity));
			    $shoppingCartItemID = db()->lastInsertId();
			    
			    // Add variables
			    if (count($product["variables"]) > 0) {
				    $insertShoppingCartProductVariableValues = db()->prepare("INSERT INTO ShoppingCartProductVariableValues (cartProductID, variableID, value) VALUES (?, ?, ?)");
				    foreach ($product["variables"] as $variable) {
					    $insertShoppingCartProductVariableValues->execute(array($shoppingCartItemID, $variable["id"], sanitize(input('variable_' . $variable["id"]))));
				    }
			    }
		    } else {
					// If the product already added to the cart sum posted quantity with the existing quantity
			    $quantity += $cartItem["quantity"];
					
			    // If the product already added to the cart update the quantity
			    $updateShoppingCartItem = db()->prepare("UPDATE ShoppingCartProducts SET quantity = ? WHERE id = ?");
			    $updateShoppingCartItem->execute(array($quantity, $cartItem["id"]));
		    }
	      
				if ($isJson) {
					return self::items();
				}
				redirect(url('cart.index'));
    }

    public function updateItemQuantity()
    {
        $validate = validate([
            'id' => 'required|numeric',
            'quantity' => 'required|numeric|min:1',
        ], false);
				
				if (count($validate) > 0) {
					response()->json([
						'status' => false,
						'errors' => $validate,
					]);
				}
	    
	      $quantity = input("quantity");
				
		    $shoppingCartID = auth()->user()->id();
				$shoppingCartItem = db()->prepare("SELECT * FROM ShoppingCartProducts WHERE shoppingCartID = ? AND id = ?");
				$shoppingCartItem->execute(array($shoppingCartID, input("id")));
				$shoppingCartItem = $shoppingCartItem->fetch();
				
				if (!$shoppingCartItem) {
					response()->json([
						'status' => false,
						'error' => 'error_product',
						'message' => t__("Something went wrong! Please try again later."),
					]);
				}
				
        $product = db()->prepare("SELECT P.*, C.isCumulative FROM Products P INNER JOIN ProductCategories C ON P.categoryID = C.id WHERE P.id = ? AND P.isActive = ?");
        $product->execute(array($shoppingCartItem["productID"], 1));
        $product = $product->fetch();
				
				if (!$product) {
					response()->json([
						'status' => false,
						'error' => 'error_product',
						'message' => t__("Something went wrong! Please try again later."),
					]);
				}
				
				// Is disabled quantity
		    if (($product["isCumulative"] == '1' || $product["disableQuantity"] == '1') && $quantity > 1) {
			    response()->json([
				    'status' => false,
				    'error' => 'error_disabled_quantity',
				    'message' => t__("You can only purchase one of this product!"),
			    ]);
		    }
				
		    // Stock Check
		    if ($product["stock"] != -1) {
			    $totalQuantityQuery = db()->prepare("SELECT SUM(quantity) FROM ShoppingCartProducts WHERE shoppingCartID = ? AND productID = ? AND id != ?");
			    $totalQuantityQuery->execute([$shoppingCartID, $product["id"], $shoppingCartItem["id"]]);
			    $totalQuantity = (int) $totalQuantityQuery->fetch(PDO::FETCH_COLUMN);
			    $totalQuantity += $quantity;
			    
			    if ($totalQuantity > $product["stock"]) {
				    response()->json([
					    'status' => false,
					    'error' => 'error_stock',
					    'message' => t__("Not enough items in the stock!"),
				    ]);
			    }
		    }
		    
		    $updateShoppingCartItem = db()->prepare("UPDATE ShoppingCartProducts SET quantity = ? WHERE id = ?");
		    $updateShoppingCartItem->execute(array($quantity, $shoppingCartItem["id"]));
	    
	      self::items();
    }

    public function deleteItem()
    {
	      $validate = validate([
            'id' => 'required|numeric',
        ]);
	    
		    if (count($validate) > 0) {
			    response()->json([
				    'status' => false,
				    'errors' => $validate,
			    ]);
		    }
	    
		    $shoppingCartID = auth()->user()->id();
		    $shoppingCartItem = db()->prepare("SELECT * FROM ShoppingCartProducts WHERE shoppingCartID = ? AND id = ?");
		    $shoppingCartItem->execute(array($shoppingCartID, input("id")));
		    $shoppingCartItem = $shoppingCartItem->fetch();
		    
		    if (!$shoppingCartItem) {
			    response()->json([
				    'status' => false,
				    'error' => 'error_product',
				    'message' => t__("Something went wrong! Please try again later."),
			    ]);
		    }
	    
        CartService::deleteItem($shoppingCartItem["id"]);
	    
	      self::items();
    }

    public function applyCoupon()
    {
	      $validate = validate([
            'coupon' => 'required',
        ]);
	    
		    if (count($validate) > 0) {
			    response()->json([
				    'status' => false,
				    'errors' => $validate,
			    ]);
		    }
	    
	      $shoppingCartID = auth()->user()->id();
        $shoppingCartProducts = CartService::getItems($shoppingCartID);
        $couponStatus = CartService::checkCoupon(input("coupon"), $shoppingCartProducts);
				
        if (!$couponStatus["status"]) {
	        response()->json([
		        'status' => false,
		        'error' => $couponStatus["error"],
		        'message' => $couponStatus["message"],
	        ]);
        }
	    
		    $couponID = $couponStatus["data"]["id"];
		    $updateShoppingCartCoupon = db()->prepare("UPDATE ShoppingCarts SET couponID = ? WHERE accountID = ?");
		    $updateShoppingCartCoupon->execute(array($couponID, auth()->user()->id()));
	    
	      self::items();
    }

    public function removeCoupon()
    {
	      $shoppingCartID = auth()->user()->id();
        $updateShoppingCartCoupon = db()->prepare("UPDATE ShoppingCarts SET couponID = ? WHERE accountID = ?");
        $updateShoppingCartCoupon->execute([null, $shoppingCartID]);
				
				self::items();
    }
	
		public function applyCreatorCode()
		{
			$validate = validate([
				'creatorCode' => 'required',
			]);
			
			if (count($validate) > 0) {
				response()->json([
					'status' => false,
					'errors' => $validate,
				]);
			}
			
			$shoppingCartID = auth()->user()->id();
			$creatorCodeStatus = CartService::checkCreatorCode(input("creatorCode"));
			
			if (!$creatorCodeStatus["status"]) {
				response()->json([
					'status' => false,
					'error' => $creatorCodeStatus["error"],
					'message' => $creatorCodeStatus["message"],
				]);
			}
			
			$creatorCodeID = $creatorCodeStatus["data"]["id"];
			$updateShoppingCartCoupon = db()->prepare("UPDATE ShoppingCarts SET creatorCodeID = ? WHERE accountID = ?");
			$updateShoppingCartCoupon->execute(array($creatorCodeID, $shoppingCartID));
			
			self::items();
		}
		
		public function removeCreatorCode()
		{
			$shoppingCartID = auth()->user()->id();
			$updateShoppingCartCreatorCode = db()->prepare("UPDATE ShoppingCarts SET creatorCodeID = ? WHERE accountID = ?");
			$updateShoppingCartCreatorCode->execute([null, $shoppingCartID]);
			
			self::items();
		}

    public function pay()
    {
			validate([
				'paymentGatewayID' => 'required',
			]);
			
			$cart = CartService::getCart();
			
			$total = $cart["total_raw"];
			$subtotal = $cart["subtotal_raw"];
			$discount = $cart["discount_raw"];
			$cashback = $cart["cashback_raw"];
			$tax = $cart["tax_raw"];
	    $gatewayFee = 0;
	  
			if (modules('store')->settings('showTermsCheckbox') == 1 && !isset($_POST["terms"])) {
		    return back()->flash('error', t__('You must accept the terms and conditions!'));
	    }
	    if ($total < modules('store')->settings('minPay')) {
		    return back()->flash('error', t__('You must have a minimum of %amount% credits in your account to use this coupon!', ['%amount%' => modules('store')->settings('minPay')]));
	    }
	    if ($total > modules('store')->settings('maxPay')) {
		    return back()->flash('error', t__('You can use a maximum of %amount% credits in your account to use this coupon!', ['%amount%' => modules('store')->settings('maxPay')]));
	    }
			if ($total <= 0) {
				return back();
			}
	  
	    $useBalance =  0;
			if (isset($_POST["useBalance"])) {
				$useBalance = (auth()->user()->credits() >= $total) ? $total : auth()->user()->credits();
			}
			
			// Pay with credits and deliver
	    if ($useBalance >= $total || modules('store')->settings('isCreditRequiredOnPurchase') == 1) {
		    if (modules('store')->settings('isCreditRequiredOnPurchase') == 1) {
			    $useBalance = $total;
			    if (auth()->user()->credits() < $total) {
						return back()->flash('error', t__("You don't have enough credit."));
			    }
		    }
				
		    $updateCredit = db()->prepare("UPDATE Accounts SET credit = credit - ? WHERE id = ?");
		    $updateCredit->execute(array($useBalance, auth()->user()->id()));
		    $insertCreditHistory = db()->prepare("INSERT INTO CreditHistory (accountID, type, price, creationDate) VALUES (?, ?, ?, ?)");
		    $insertCreditHistory->execute(array(auth()->user()->id(), 2, $useBalance, datetime()));
		    
		    if ($cashback > 0) {
			    $updateCredit = db()->prepare("UPDATE Accounts SET credit = credit + ? WHERE id = ?");
			    $updateCredit->execute(array($cashback, auth()->user()->id()));
			    $insertCreditHistory = db()->prepare("INSERT INTO CreditHistory (accountID, type, price, creationDate) VALUES (?, ?, ?, ?)");
			    $insertCreditHistory->execute(array(auth()->user()->id(), 1, $cashback, datetime()));
		    }
		    
		    $createOrder = db()->prepare("INSERT INTO Orders (accountID, coupon, total, discount, subtotal, tax, gatewayFee, paymentID, paymentAPI, status, type, credit, earnings, cashback, creationDate) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
		    $createOrder->execute(array(
			    auth()->user()->id(),
			    $cart["coupon"] ? $cart["coupon"]["code"] : null,
			    $total,
			    $discount,
			    $subtotal,
					$tax,
			    $gatewayFee,
			    "",
			    'credit',
			    1,
			    1,
			    $useBalance,
			    0,
			    $cashback,
			    datetime()
		    ));
		    $orderID = db()->lastInsertId();
		    foreach ($cart["items"] as $cartItem) {
		      $createOrderProducts = db()->prepare("INSERT INTO OrderProducts (orderID, productID, quantity, unitPrice) VALUES (?, ?, ?, ?)");
			    $createOrderProducts->execute(array($orderID, $cartItem["product_id"], $cartItem["quantity"], $cartItem["discounted_price_raw"] == 0 ? $cartItem["price_raw"] : $cartItem["discounted_price_raw"]));
			    $orderProductID = db()->lastInsertId();
			    
			    foreach ($cartItem["variables"] as $variable) {
		        $insertOrderProductVariableValues = db()->prepare("INSERT INTO OrderProductVariableValues (orderProductID, variableID, value) VALUES (?, ?, ?)");
				    $insertOrderProductVariableValues->execute(array($orderProductID, $variable["id"], $variable["value"]));
			    }
		    }
		    if ($cart["coupon"] != null) {
			    $insertProductCouponsHistory = db()->prepare("INSERT INTO ProductCouponsHistory (accountID, couponID, productID, creationDate) VALUES (?, ?, ?, ?)");
			    $insertProductCouponsHistory->execute(array(auth()->user()->id(), $cart["coupon"]["id"], $orderID, datetime()));
		    }
		    if ($cart["creator_code"] != null) {
			    $insertOrderCreatorCodes = db()->prepare("INSERT INTO OrderCreatorCodes (orderID, codeID, discount, revenueShare) VALUES (?, ?, ?, ?)");
			    $insertOrderCreatorCodes->execute(array($orderID, $cart["creator_code"]["id"], $cart["creator_code"]["discount_raw"], $cart["creator_code"]["revenue_share_raw"]));
			    
			    // Add revenue share to the creator
			    $creatorCodeAccount = db()->prepare("SELECT id FROM Accounts WHERE id = ?");
			    $creatorCodeAccount->execute(array($cart["creator_code"]["creator_id"]));
			    $creatorCodeAccount = $creatorCodeAccount->fetch(PDO::FETCH_ASSOC);
			    
			    if ($creatorCodeAccount) {
				    $updateCreatorCodeAccount = db()->prepare("UPDATE Accounts SET credit = credit + ? WHERE id = ?");
				    $updateCreatorCodeAccount->execute(array($cart["creator_code"]["revenue_share_raw"], $cart["creator_code"]["creator_id"]));
				    $insertCreditHistory = db()->prepare("INSERT INTO CreditHistory (accountID, type, price, creationDate) VALUES (?, ?, ?, ?)");
				    $insertCreditHistory->execute(array($cart["creator_code"]["creator_id"], 1, $cart["creator_code"]["revenue_share_raw"], datetime()));
			    }
		    }
		    
		    foreach ($cart["items"] as $cartItem) {
			    if (modules('chest')->isActive()) {
				    for ($i = 0; $i < $cartItem["quantity"]; $i++) {
					    $insertChests = db()->prepare("INSERT INTO Chests (accountID, productID, canGift, status, creationDate) VALUES (?, ?, ?, ?, ?)");
					    $insertChests->execute(array(auth()->user()->id(), $cartItem["product_id"], ($cartItem["can_gift"] ? "1" : "0"), 0, datetime()));
					    $chestID = db()->lastInsertId();
					    
					    $insertChestProductVariableValues = db()->prepare("INSERT INTO ChestProductVariableValues (chestID, variableID, value) VALUES (?, ?, ?)");
					    foreach ($cartItem["variables"] as $variable) {
						    $insertChestProductVariableValues->execute(array($chestID, $variable["id"], $variable["value"]));
					    }
				    }
			    }
			    
			    if ($cartItem["stock"] != -1) {
				    $updateStock = db()->prepare("UPDATE Products SET stock = stock - ? WHERE id = ?");
				    $updateStock->execute(array($cartItem["quantity"], $cartItem["product_id"]));
			    }
		    }
		    
		    if (!modules('chest')->isActive()) {
					$cartItemsForDelivery = [];
					foreach ($cart["items"] as $cartItem) {
						$variablesForDelivery = [];
						foreach ($cartItem["variables"] as $variable) {
							$variablesForDelivery[$variable["identifier"]] = $variable["value"];
						}
						
						$cartItemsForDelivery[] = [
							'id' => $cartItem["product_id"],
							'quantity' => $cartItem["quantity"],
							'variables' => $variablesForDelivery,
						];
					}
			    DeliveryService::deliver($cartItemsForDelivery, auth()->user()->id(), "order", $orderID);
		    }
		    
		    $deleteShoppingCartProducts = db()->prepare("DELETE FROM ShoppingCartProducts WHERE shoppingCartID = ?");
		    $deleteShoppingCartProducts->execute(array(auth()->user()->id()));
				
				$deleteShoppingCartCoupon = db()->prepare("UPDATE ShoppingCarts SET couponID = ? AND creatorCodeID =? WHERE accountID = ?");
				$deleteShoppingCartCoupon->execute([null, null, auth()->user()->id()]);
		    
		    if (modules('store')->settings('sendEmailOnPurchase') != '0') {
					$orderProductsArray = [];
					foreach ($cart["items"] as $cartItem) {
						$orderProductsArray[] = new Product($cartItem["name"], $cartItem["discounted_price_raw"] == 0 ? $cartItem["price_raw"] : $cartItem["discounted_price_raw"], $cartItem["quantity"]);
					}
					$orderObject = new Order(
						$orderID,
						"product",
						$total,
						$subtotal,
						$tax,
						$gatewayFee,
						$discount,
						$useBalance,
						$cart["coupon"] ? $cart["coupon"]["code"] : null,
						$orderProductsArray
					);
			    JobDispatcher::dispatch((new SendEmail(
						auth()->user()->email(),
						modules('store')->settings('purchaseEmailTitle'),
				    EmailService::processPurchaseEmailContent(auth()->user()->displayName(), $orderObject)
			    )));
		    }
		    
		    foreach ($cart["items"] as $cartItem) {
			    JobDispatcher::dispatch((new SendDiscordWebhook('store.purchase.product', [
				    "username" => auth()->user()->displayName(),
				    "product" => $cartItem["name"],
				    "category" => $cartItem["category"],
				    "quantity" => $cartItem["quantity"],
				    "price" => $cartItem["discounted_price_raw"] == 0 ? $cartItem["price_raw"] : $cartItem["discounted_price_raw"],
				    "currency" => settings('currency'),
			    ])));
		    }
		    
		    redirect(url('payment.success'));
	    }
	    else {
		    // Only pay with credit requirement
		    if (modules('store')->settings('isCreditRequiredOnPurchase') == 1) {
			    redirect(url('payment.error'));
		    }
		    
		    $paymentGateway = db()->prepare("SELECT * FROM PaymentGateways WHERE slug = ? AND status = ?");
		    $paymentGateway->execute([input("paymentGatewayID"), 1]);
		    $paymentGateway = $paymentGateway->fetch();
		    if (!$paymentGateway) return back();
		    
		    if ($paymentGateway["feeStatus"] == 1) {
			    $gatewayFee = ($total - $useBalance) * ($paymentGateway["transactionFee"] / 100);
			    if ($paymentGateway["fixedFee"] > 0) {
				    $gatewayFee += $paymentGateway["fixedFee"];
			    }
		    }
				$total += $gatewayFee;
		    
		    $createOrder = db()->prepare("INSERT INTO Orders (accountID, coupon, total, discount, subtotal, tax, gatewayFee, paymentID, paymentAPI, status, type, credit, earnings, cashback, creationDate) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
		    $createOrder->execute(array(
			    auth()->user()->id(),
			    $cart["coupon"] ? $cart["coupon"]["code"] : null,
			    $total,
			    $discount,
			    $subtotal,
					$tax,
			    $gatewayFee,
			    "",
			    $paymentGateway["slug"],
			    0,
			    1,
			    $useBalance,
			    $total-$useBalance,
			    $cashback,
			    datetime()
		    ));
		    $orderID = db()->lastInsertId();
		    foreach ($cart["items"] as $cartItem) {
		      $createOrderProducts = db()->prepare("INSERT INTO OrderProducts (orderID, productID, quantity, unitPrice) VALUES (?, ?, ?, ?)");
			    $createOrderProducts->execute(array($orderID, $cartItem["product_id"], $cartItem["quantity"], $cartItem["discounted_price_raw"] == 0 ? $cartItem["price_raw"] : $cartItem["discounted_price_raw"]));
			    $orderProductID = db()->lastInsertId();
			    
			    foreach ($cartItem["variables"] as $variable) {
				    $insertOrderProductVariableValues = db()->prepare("INSERT INTO OrderProductVariableValues (orderProductID, variableID, value) VALUES (?, ?, ?)");
				    $insertOrderProductVariableValues->execute(array($orderProductID, $variable["id"], $variable["value"]));
			    }
		    }
		    if ($cart["creator_code"] != null) {
			    $insertOrderCreatorCodes = db()->prepare("INSERT INTO OrderCreatorCodes (orderID, codeID, discount, revenueShare) VALUES (?, ?, ?, ?)");
			    $insertOrderCreatorCodes->execute(array($orderID, $cart["creator_code"]["id"], $cart["creator_code"]["discount_raw"], $cart["creator_code"]["revenue_share_raw"]));
		    }
		    
				$customer = PaymentGatewayService::getCustomer();
				
				// Order object
		    $products = [];
		    foreach ($cart["items"] as $cartItem) {
			    $products[] = new Product($cartItem["name"], $cartItem["discounted_price_raw"] == 0 ? $cartItem["price_raw"] : $cartItem["discounted_price_raw"], $cartItem["quantity"]);
		    }
		    $order = new Order(
			    $orderID,
			    "product",
			    $total,
			    $subtotal,
			    $tax,
			    $gatewayFee,
			    $discount,
			    $useBalance,
			    $cart["coupon"] ? $cart["coupon"]["code"] : null,
			    $products
		    );
		    
		    try {
			    $payment = PaymentGatewayService::init($paymentGateway["slug"], $paymentGateway["currency"], $paymentGateway["config"]);
					$payment->setCustomer($customer);
					$payment->setOrder($order);
			    $payment->pay();
		    } catch (Exception $e) {
			    if (settings('debugModeStatus') == 1) {
				    return $e->getMessage();
			    } else {
				    return $this->view('payment.error');
			    }
		    }
	    }
    }
		
		public function featuredProducts()
		{
			if (modules('store')->settings('featuredProducts') == 0) {
				response()->json([]);
			}
			
			$featuredProducts = db()->prepare("SELECT P.id, P.name, P.price, P.discountedPrice, P.discountExpiryDate, P.stock, P.imageID, P.imageType FROM Products P WHERE P.isFeaturedProduct = ? AND P.stock != ? AND P.isActive = ? ORDER BY P.priority DESC");
			$featuredProducts->execute([1, 0, 1]);
			$featuredProductsArray = $featuredProducts->fetchAll(PDO::FETCH_ASSOC);
			$response = [];
			for ($i = 0; $i < count($featuredProductsArray); $i++) {
				$featuredProductsArray[$i]["image"] = "/assets/core/images/store/products/".$featuredProductsArray[$i]["imageID"].".".$featuredProductsArray[$i]["imageType"];
				unset($featuredProductsArray[$i]["imageID"]);
				unset($featuredProductsArray[$i]["imageType"]);
				
				$discountProducts = explode(",", modules('store')->settings('discountProducts'));
				$discountedPriceStatus = ($featuredProductsArray[$i]["discountedPrice"] != 0 && ($featuredProductsArray[$i]["discountExpiryDate"] > datetime() || $featuredProductsArray[$i]["discountExpiryDate"] == '1000-01-01 00:00:00'));
				$storeDiscountStatus = (modules('store')->settings('discount') != 0 && (in_array($featuredProductsArray[$i]["id"], $discountProducts) || modules('store')->settings('discountProducts') == '0') && (modules('store')->settings('discountExpiryDate') > datetime() || modules('store')->settings('discountExpiryDate') == '1000-01-01 00:00:00'));
				
				if ($discountedPriceStatus || $storeDiscountStatus) {
					$discountedPrice = $storeDiscountStatus ? (($featuredProductsArray[$i]["price"]*(100- modules('store')->settings('discount')))/100) : $featuredProductsArray[$i]["discountedPrice"];
					$featuredProductsArray[$i]["discountPercent"] = (($storeDiscountStatus) ? modules('store')->settings('discount') : round((($featuredProductsArray[$i]["price"]-$featuredProductsArray[$i]["discountedPrice"])*100)/($featuredProductsArray[$i]["price"])));
					$featuredProductsArray[$i]["discountedPrice"] = money()->format($discountedPrice);
				} else {
					$featuredProductsArray[$i]["discountPercent"] = 0;
					$featuredProductsArray[$i]["discountedPrice"] = 0;
				}
				$featuredProductsArray[$i]["price"] = money()->format($featuredProductsArray[$i]["price"]);
				$featuredProductsArray[$i]["rawPrice"] = money()->formatDecimal($featuredProductsArray[$i]["price"]);
				unset($featuredProductsArray[$i]["discountExpiryDate"]);
				
				$response[] = $featuredProductsArray[$i];
			}
			
			response()->json($response);
		}
}