<?php
	namespace Main\Libs;
	/* LeaderOS
	 * Discord OAuth Class
	 * @author : benfiratkaya
	 * @copyright : https://firatkaya.net
	 */
	
	use Exception;
	use Main\Services\AvatarService;
	
	class Discord
	{
		private string $baseUrl = "https://discord.com";
		private string $clientID;
		private string $clientSecret;
		private string $botToken;
		
		public function __construct($clientID, $clientSecret, $botToken = null)
		{
			$this->clientID = $clientID;
			$this->clientSecret = $clientSecret;
			$this->botToken = $botToken;
		}
		
		/**
		 * Get Discord Login URL
		 *
		 * @param $redirectUrl
		 * @param array $scopes
		 * @return string
		 */
		public function getLoginUrl($redirectUrl, array $scopes = ['identify', 'email', 'guilds', 'guilds.join']): string
		{
			$params = array(
				'client_id' => $this->clientID,
				'redirect_uri' => $redirectUrl,
				'response_type' => 'code',
				'scope' => implode(' ', $scopes)
				//'state' => "1234567890"
			);
			return $this->baseUrl . "/api/oauth2/authorize?" . http_build_query($params);
		}
		
		/**
		 * Get Access Token
		 * @param $code
		 * @param $redirectUrl
		 * @throws Exception
		 */
		public function getAccessToken($code, $redirectUrl) {
			$params = [
				'client_id' => $this->clientID,
				'client_secret' => $this->clientSecret,
				'grant_type' => 'authorization_code',
				'code' => $code,
				'redirect_uri' => $redirectUrl
			];
			$response = $this->request("/api/oauth2/token", 'POST', $params);
			return $response->access_token;
		}
		
		/**
		 * Get User
		 *
		 * @param $accessToken
		 * @return mixed
		 * @throws Exception
		 */
		public function getUser($accessToken) {
			return $this->request("/api/users/@me", 'GET', [], $this->createAccessTokenHeader($accessToken));
		}
		
		public function joinGuild($accessToken, $guildID, $memberID) {
			return $this->request("/api/guilds/$guildID/members/$memberID", 'PUT', [
				'access_token' => $accessToken
			], $this->creatBotTokenHeader());
		}
		
		public function giveRole($guildID, $memberID, $roleID) {
			return $this->request("/api/guilds/$guildID/members/$memberID/roles/$roleID", 'PUT',
				[],
				$this->creatBotTokenHeader());
		}
		
		public function removeRole($guildID, $memberID, $roleID) {
			return $this->request("/api/guilds/$guildID/members/$memberID/roles/$roleID", 'DELETE',
				[],
				$this->creatBotTokenHeader());
		}
		
		// Create support ticket
		public function createTicket(
			$discordGuildID,
			$discordCategoryID,
			$ticketID,
			$customFields = [],
			$category,
			$title,
			$message,
			$username,
			$authorID,
			$everyoneRoleID,
			$staffRoleID,
			$ticketEmbedDetailsTemplate
		)
		{
			$permissionOverwrites = [
				[
					'id' => $everyoneRoleID,
					'type' => 0,
					'deny' => '1024',
				],
				[
					'id' => $staffRoleID,
					'type' => 0,
					'allow' => '3072',
				]
			];
			if ($authorID !== null) {
				$permissionOverwrites[] = [
					'id' => $authorID,
					'type' => 1,
					'allow' => '3072',
				];
			}
			$channelRequest = $this->request("/api/v9/guilds/$discordGuildID/channels", 'POST', [
				'name' => 'ticket-' . $ticketID,
				'type' => 0,
				'topic' => $title,
				'parent_id' => $discordCategoryID,
				'permission_overwrites' => $permissionOverwrites
			], $this->creatBotTokenHeader());
			
			$channelID = $channelRequest->id;
			
			$messageContent = preg_replace("/\r\n/s", "", $ticketEmbedDetailsTemplate);
			
			$messageContent = str_replace([
				'{category}',
				'{username}',
				'{message}'
			], [
				$category,
				$username,
				$message
			], $messageContent);
			
			if (count($customFields) > 0) {
				preg_match('/%fields%(.*?)%endfields%/s', $messageContent, $matches);
				$replacement = "";
				foreach ($customFields as $customField) {
					$replacement .= str_replace(
						['{field}', '{fieldValue}'],
						[$customField["field"], $customField["value"]],
						$matches[0]
					);
				}
				$messageContent = preg_replace('/%fields%(.*?)%endfields%/s', $replacement, $messageContent);
				$messageContent = str_replace(
					['%fields%', '%endfields%'],
					['', ''],
					$messageContent
				);
			}
			else {
				$messageContent = preg_replace('/%fields%(.*?)%endfields%/s', '', $messageContent);
			}
			
			$messageContent = preg_replace('/{newline}/s', "\n", $messageContent);
			
			$messageRequest = $this->request("/api/v9/channels/$channelID/messages", 'POST', [
				'embeds' => [[
					'author' => [
						'name' => $title,
					],
					'description' => $messageContent,
					"color" => 1632000,
				]],
				'components' => [[
					'type' => 1,
					'components' => [[
						'type' => 2,
						'style' => 2,
						'label' => 'Close',
						'emoji' => [
							'name' => '❌'
						],
						'custom_id' => 'tickets_closeBtn'
					]]
				]]
			], $this->creatBotTokenHeader());
			
			return $channelRequest;
		}
		
		// Send message to support ticket
		public function sendTicketMessage(
			$discordChannelID,
			$userID,
			$username,
			$message,
			$isStaff = false
		)
		{
			$messageRequest = $this->request("/api/v9/channels/$discordChannelID/messages", 'POST', [
				'embeds' => [[
					'author' => [
						'name' => $username,
						'icon_url' => AvatarService::get($userID, $username),
					],
					'description' => $message,
					"color" => $isStaff ? 15548997 : 3447003,
				]],
			], $this->creatBotTokenHeader());
			
			return $messageRequest;
		}
		
		/**
		 * Request
		 *
		 * @param string $url URL to fetch
		 * @param string $method HTTP method
		 * @param array $data
		 * @param array $headers Headers to send
		 * @return mixed
		 * @throws Exception
		 */
		public function request(string $url, string $method = 'GET', array $data = [], array $headers = [])
		{
			$url = $this->baseUrl . $url;
			$request = curl_init($url);
			
			// Curl options
			curl_setopt($request, CURLOPT_RETURNTRANSFER, true);
			curl_setopt($request, CURLOPT_CUSTOMREQUEST, $method);
			
			// Custom headers
			if (!empty($headers)) {
				curl_setopt($request, CURLOPT_HTTPHEADER, $this->createHeaders($headers));
				
				foreach ($headers as $key => $value) {
					if ($key === 'Content-Type' && $value === 'application/json') {
						$data = json_encode($data);
						break;
					}
				}
			}
			
			// Custom Data
			curl_setopt($request, CURLOPT_POSTFIELDS, $data);
			
			$response = curl_exec($request);
			$error = curl_error($request);
			$errorNo = curl_errno($request);
			curl_close($request);
			
			if ($errorNo !== 0)
				throw new Exception($error, $errorNo);
			
			return json_decode($response);
		}
		
		private function createAccessTokenHeader($token): array
		{
			return [
				'Authorization' => "Bearer " . $token
			];
		}
		
		private function creatBotTokenHeader(): array
		{
			return [
				'Content-Type' => 'application/json',
				'Authorization' => "Bot " . $this->botToken
			];
		}
		
		// Creates curl headers from an array
		private function createHeaders($headers): array
		{
			$result = [];
			foreach ($headers as $key => $value) {
				$result[] = $key . ': ' . $value;
			}
			return $result;
		}
	}