<?php
	namespace Main\Libs\Payment;
	
	use Main\Libs\Payment\Exceptions\UnknownCustomerException;
	
	abstract class PaymentGateway
	{
		protected $name, $currency, $currencyRate, $config, $lang, $sucessUrl, $cancelUrl, $callbackUrl;
		protected bool $isBrowserSideCallback = false;
		protected bool $isContactRequired = false;
		protected Customer $customer;
		protected Order $order;
		
		public function __construct($name, $currency, $config)
		{
			$this->name = $name;
			$this->currency = $currency;
			$this->currencyRate = self::getCurrencyRate($currency);
			$this->config = $config;
			$this->lang = lang();
			$this->sucessUrl = website_url(url('payment.success'));
			$this->cancelUrl = website_url(url('payment.error'));
			$this->callbackUrl = website_url(url('payment.callback', ['provider' => $name]));
		}
		
		public function setCustomer(Customer $customer)
		{
			$this->customer = $customer;
		}
		
		public function setOrder(Order $order)
		{
			// Update the order values with the currency rate
			$order->total = money()->formatDecimal($order->total * $this->currencyRate);
			$order->subtotal = money()->formatDecimal($order->subtotal * $this->currencyRate);
			$order->discount = money()->formatDecimal($order->discount * $this->currencyRate);
			$order->credits = money()->formatDecimal($order->credits * $this->currencyRate);
			$order->paymentAmount = money()->formatDecimal($order->paymentAmount * $this->currencyRate);
			
			foreach ($order->products as $product) {
				$product->price = money()->formatDecimal($product->price * $this->currencyRate);
				$product->total = money()->formatDecimal($product->total * $this->currencyRate);
			}
			
			$this->order = $order;
		}
		
		public function getCustomer(): Customer
		{
			return $this->customer;
		}
		
		public function getOrder(): Order
		{
			return $this->order;
		}
		
		/**
		 * @throws UnknownCustomerException
		 */
		public function pay()
		{
			if ($this->isContactRequired() && ($this->customer->contact == null)) {
				throw new UnknownCustomerException($this->order->type);
			}
			
			$this->payment();
		}
		
		abstract public function payment();
		abstract public function callback(): SuccessfulPayment;
		
		public function getName()
		{
			return $this->name;
		}
		
		public function getConfig()
		{
			return $this->config;
		}
		
		public function getSuccessUrl(): string
		{
			return $this->sucessUrl;
		}
		
		public function getCancelUrl(): string
		{
			return $this->cancelUrl;
		}
		
		public function getCallbackUrl(): string
		{
			return $this->callbackUrl;
		}
		
		public function setSuccessUrl($url)
		{
			$this->sucessUrl = $url;
		}
		
		public function setCancelUrl($url)
		{
			$this->cancelUrl = $url;
		}
		
		public function setCallbackUrl($url)
		{
			$this->callbackUrl = $url;
		}
		
		public function isBrowserSideCallback(): bool
		{
			return $this->isBrowserSideCallback;
		}
		
		public function setBrowserSideCallback($isBrowserSideCallback)
		{
			$this->isBrowserSideCallback = $isBrowserSideCallback;
		}
		
		public function isContactRequired(): bool
		{
			return $this->isContactRequired;
		}
		
		public function setContactRequired($isContactRequired)
		{
			$this->isContactRequired = $isContactRequired;
		}
		
		public function getCurrency()
		{
			return $this->currency;
		}
		
		public function setCurrency($currency)
		{
			$this->currency = $currency;
		}
		
		private function getCurrencyRate($currency)
		{
			$currencies = money()->getCurrencies();
			foreach ($currencies as $item) {
				if ($item->getCode() == $currency) {
					return $item->getRate();
				}
			}
			
			return 1;
		}
	}