<?php
	
	namespace Main\Services;
	
	use Exception;
	
	class ImageCropService
	{
		/**
		 * @throws Exception
		 */
		public static function processAvatar(string $sourcePath, int $size): void
		{
			self::resizeAndCropImage($sourcePath, $size, $size);
		}
		
		/**
		 * @throws Exception
		 */
		private static function resizeAndCropImage(string $sourcePath, int $xSize = 256, int $ySize = 256): void
		{
			// Get image type
			$imageInfo = getimagesize($sourcePath);
			if ($imageInfo === false) {
				throw new Exception('Could not get image information');
			}
			
			// If it's a GIF, use Imagick to preserve animation
			if ($imageInfo[2] === IMAGETYPE_GIF) {
				self::resizeAndCropGifWithImagick($sourcePath, $xSize, $ySize);
				return;
			}
			
			// If Imagick is installed, use it
			if (class_exists('Imagick')) {
				self::resizeAndCropWithImagick($sourcePath, $xSize, $ySize);
				return;
			}
			
			// Use GD if Imagick is not installed
			self::resizeAndCropWithGD($sourcePath, $xSize, $ySize);
		}
		
		/**
		 * @throws Exception
		 */
		private static function resizeAndCropGifWithImagick(string $sourcePath, int $xSize, int $ySize): void
		{
			if (!class_exists('Imagick')) {
				throw new Exception('GIF cropping requires Imagick extension, which is not installed on this server.');
			}
			
			try {
				$imagick = new \Imagick($sourcePath);
				
				// Split gif animation into image
				$imagick = $imagick->coalesceImages();
				
				// Crop images one by one
				do {
					// Crop and resize image
					$imagick->cropThumbnailImage($xSize, $ySize);
					
					// Reset page geometry
					$imagick->setImagePage($xSize, $ySize, 0, 0);
					
				} while ($imagick->nextImage());
				
				// Reassemble the animation
				$imagick = $imagick->deconstructImages();
				
				// Save gif animation
				$imagick->writeImages($sourcePath, true);
				
				// Clean up
				$imagick->clear();
			} catch (\ImagickException $e) {
				throw new Exception('An error occurred while cropping GIF: ' . $e->getMessage());
			}
		}
		
		/**
		 * @throws Exception
		 */
		private static function resizeAndCropWithImagick(string $sourcePath, int $xSize, int $ySize): void
		{
			try {
				$imagick = new \Imagick($sourcePath);
				
				// Crop and resize image
				$imagick->cropThumbnailImage($xSize, $ySize);
				
				// Save image
				$imagick->writeImages($sourcePath, true);
				
				// Clean up
				$imagick->clear();
			} catch (\ImagickException $e) {
				throw new Exception('An error occurred while cropping image: ' . $e->getMessage());
			}
		}
		
		/**
		 * @throws Exception
		 */
		private static function resizeAndCropWithGD(string $sourcePath, int $xSize, int $ySize): void
		{
			// Get original image dimensions
			[$originalWidth, $originalHeight, $imageType] = getimagesize($sourcePath);
			
			// Supported image types
			$imageTypes = [
				IMAGETYPE_JPEG => 'imagecreatefromjpeg',
				IMAGETYPE_PNG => 'imagecreatefrompng',
				IMAGETYPE_GIF => 'imagecreatefromgif',
				IMAGETYPE_WEBP => 'imagecreatefromwebp'
			];
			
			if (!isset($imageTypes[$imageType])) {
				throw new Exception('Unsupported image type');
			}
			
			// Create original image
			$createFunc = $imageTypes[$imageType];
			$sourceImage = $createFunc($sourcePath);
			
			if (!$sourceImage) {
				throw new Exception('Failed to create image from source');
			}
			
			// Calculate new size (determine maximum square area)
			$minSize = min($originalWidth, $originalHeight);
			$cropX = ($originalWidth - $minSize) / 2;
			$cropY = ($originalHeight - $minSize) / 2;
			
			// Perform center crop
			$croppedImage = imagecreatetruecolor($minSize, $minSize);
			
			// Preserve transparency for PNG and GIF
			if ($imageType === IMAGETYPE_PNG || $imageType === IMAGETYPE_GIF) {
				imagealphablending($croppedImage, false);
				imagesavealpha($croppedImage, true);
				$transparent = imagecolorallocatealpha($croppedImage, 255, 255, 255, 127);
				imagefilledrectangle($croppedImage, 0, 0, $minSize, $minSize, $transparent);
			}
			
			imagecopyresampled($croppedImage, $sourceImage, 0, 0, $cropX, $cropY, $minSize, $minSize, $minSize, $minSize);
			
			// Resize to target dimensions
			$resizedImage = imagecreatetruecolor($xSize, $ySize);
			
			// Preserve transparency for PNG and GIF
			if ($imageType === IMAGETYPE_PNG || $imageType === IMAGETYPE_GIF) {
				imagealphablending($resizedImage, false);
				imagesavealpha($resizedImage, true);
				$transparent = imagecolorallocatealpha($resizedImage, 255, 255, 255, 127);
				imagefilledrectangle($resizedImage, 0, 0, $xSize, $ySize, $transparent);
			}
			
			imagecopyresampled($resizedImage, $croppedImage, 0, 0, 0, 0, $xSize, $ySize, $minSize, $minSize);
			
			// Save the image
			$success = false;
			switch ($imageType) {
				case IMAGETYPE_JPEG:
					$success = imagejpeg($resizedImage, $sourcePath, 90);
					break;
				case IMAGETYPE_PNG:
					$success = imagepng($resizedImage, $sourcePath, 9);
					break;
				case IMAGETYPE_GIF:
					$success = imagegif($resizedImage, $sourcePath);
					break;
				case IMAGETYPE_WEBP:
					$success = imagewebp($resizedImage, $sourcePath, 90);
					break;
			}
			
			if (!$success) {
				throw new Exception('Failed to save image');
			}
			
			// Free memory
			imagedestroy($sourceImage);
			imagedestroy($croppedImage);
			imagedestroy($resizedImage);
		}
	}