<?php
	namespace Main\Core;
	
	use Exception;
	
	class Money
	{
		private Currency $base_currency;
		private Currency $currency;
		private array $currencies;
		
		/**
		 * @throws Exception
		 */
		public function __construct(string $base_currency_code, string $currency_code, array $currencies = [])
		{
			$this->setCurrencies($currencies);
			$this->setBaseCurrency($base_currency_code);
			$this->setCurrency($currency_code);
		}
		
		/**
		 * @throws Exception
		 */
		private function setBaseCurrency(string $base_currency_code): void
		{
			// Find the base currency from the provided list
			foreach ($this->currencies as $currency) {
				if ($currency->getCode() === $base_currency_code) {
					$this->base_currency = $currency;
					return;
				}
			}
			// Throw an exception if the base currency is not found
			throw new Exception("Base currency not found in the provided currencies list.");
		}
		
		/**
		 * @throws Exception
		 */
		public function setCurrency(string $currency_code): void
		{
			foreach ($this->currencies as $currency) {
				if ($currency->getCode() === $currency_code) {
					$this->currency = $currency;
					return;
				}
			}
			// Throw an exception if the currency is not found
			throw new Exception("Active currency not found in the provided currencies list.");
		}
		
		public function setCurrencies(array $currencies = []): void
		{
			$this->currencies = [];
			foreach ($currencies as $currency) {
				$this->currencies[] = new Currency($currency['code'], $currency['symbol'], $currency['rate'], $currency['suffix'], $currency['prefix'], $currency['label']);
			}
		}
		
		public function getCurrency(): Currency
		{
			return $this->currency;
		}
		
		public function getCurrencies(): array
		{
			return $this->currencies;
		}
		
		public function convert($amount): float
		{
			return $amount * $this->currency->getRate();
		}
		
		public function formatDecimal($amount, $raw = true): string
		{
			return number_format((float)$amount, 2, '.', $raw ? '' : ',');
		}
		
		public function removeZeroDecimal($amount): string
		{
			return substr($amount, -3) == ".00" ? substr($amount, 0, -3) : $amount;
		}
		
		public function real($amount, $convert = true): string
		{
			$amount = $convert ? $this->convert($amount) : $amount;
			$amount = $this->formatDecimal($amount, false);
			if (modules('store')->settings('removeMoneyZeroDecimal') == 1) {
				$amount = $this->removeZeroDecimal($amount);
			}
			
			// If the currency is not the base currency, add the currency symbol
			if (!$convert) {
				return $this->base_currency->getPrefix() . $amount . $this->base_currency->getSuffix();
			}
			
			return $this->currency->getPrefix() . $amount . $this->currency->getSuffix();
		}
		
		public function format($amount, $convert = true): string
		{
			if (modules('store')->settings('isCreditRequiredOnPurchase') == 1) {
				return $this->credits($amount);
			}
			
			return $this->real($amount, $convert);
		}
		
		public function credits($amount): string
		{
			$amount = $this->formatDecimal($amount, false);
			if (modules('store')->settings('removeMoneyZeroDecimal') == 1) {
				$amount = $this->removeZeroDecimal($amount);
			}
			
			if (modules('credit')->settings('creditText') != "" && modules('credit')->settings('creditText') != null)
				return $amount." ".modules('credit')->settings('creditText');
			
			return t__('%credit% credit(s)', ['%credit%' => $amount]);
		}
	}