<?php
	namespace Main\Core;
	
	class User {
		private int $id;
		private string $username;
		private string $displayName;
		private ?string $email;
		private float $credits;
		private string $avatar;
		private string $createdAt;
		private array $roles;
		private array $permissions;
		private array $supportPermissions;
		private bool $isTfaRequired;
		private array $data;
		
		public function __construct(
			int $id,
			string $username,
			string $displayName,
			?string $email,
			float $credits,
			string $avatar,
			string $createdAt,
			bool $isTfaVerified = null,
			array $roles = [],
			array $permissions = [],
			array $supportPermissions = [],
			array $data = []
		) {
			$this->id = $id;
			$this->username = $username;
			$this->displayName = $displayName;
			$this->email = $email;
			$this->credits = $credits;
			$this->avatar = $avatar;
			if (!modules('tfa')->isActive()) {
				$this->isTfaRequired = false;
			} else {
				$this->isTfaRequired = $isTfaVerified === null ? false : !$isTfaVerified;
			}
			$this->createdAt = $createdAt;
			$this->roles = $roles;
			$this->permissions = $permissions;
			$this->data = $data;
			
			if ($this->can('MANAGE_SUPPORT')) {
				$supportPermissions['STRICT_MANAGE_SUPPORT_TICKETS'] = true;
			}
			$this->supportPermissions = $supportPermissions;
		}
		
		public function id(): int
		{
			return $this->id;
		}
		
		public function username(): string
		{
			return $this->username;
		}
		
		public function displayName(): string
		{
			return $this->displayName;
		}
		
		public function email(): ?string
		{
			return $this->email;
		}
		
		public function credits(): float
		{
			return $this->credits;
		}
		
		public function avatar(): string
		{
			return $this->avatar;
		}
		
		public function createdAt(): string
		{
			return $this->createdAt;
		}
		
		public function roles(): array
		{
			return $this->roles;
		}
		
		public function roleNames(): array
		{
			return array_column($this->roles, 'name');
		}
		
		public function primaryRole(): string
		{
			return $this->roles[0];
		}
		
		public function primaryRoleName(): string
		{
			return $this->roles[0]['name'];
		}
		
		public function primaryRoleSlug(): string
		{
			return $this->roles[0]['slug'];
		}
		
		public function permissions(): array
		{
			return $this->permissions;
		}
		
		public function supportPermissions($permission = null)
		{
			return $permission === null ? $this->supportPermissions : $this->supportPermissions[$permission];
		}
		
		public function data(string $key)
		{
			return $this->data[$key];
		}
		
		public function can($perm): bool
		{
      if (is_array($perm)) {
        foreach ($perm as $p) {
          if ($this->can($p))
            return true;
        }
        return false;
      }

			if (in_array($perm, $this->permissions) || in_array('SUPER_ADMIN', $this->permissions))
				return true;
			
			return false;
		}
		
		public function isEmailVerified(): bool
		{
			return $this->data('isVerified') == 1;
		}
		
		public function isStaff(): bool
		{
			if ($this->can('VIEW_DASHBOARD'))
				return true;
			
			return false;
		}

    public function isSuperAdmin(): bool
    {
      if (in_array('SUPER_ADMIN', $this->permissions))
        return true;

      return false;
    }
		
		public function canManageSupportCategory($permission, $categoryId): bool
		{
			if ($this->supportPermissions['STRICT_MANAGE_SUPPORT_TICKETS'])
				return true;
			
			if (in_array($categoryId, $this->supportPermissions[$permission]))
				return true;
			
			return false;
		}
		
		public function isTfaRequired(): bool
		{
			return $this->isTfaRequired;
		}
	}