<?php
	
	use Carbon\Carbon;
	use Carbon\Factory;
	use Cocur\Slugify\Slugify;
	use League\CommonMark\GithubFlavoredMarkdownConverter;
	use League\Plates\Engine;
	use Main\Core\Asset;
	use Main\Core\Auth;
	use Main\Core\Cache;
	use Main\Core\Cookie;
	use Main\Core\Forum;
	use Main\Core\Module;
	use Main\Core\Money;
	use Main\Core\Redirect;
	use Main\Core\Seo;
	use Main\Core\Session;
	use Main\Core\Theme;
	use Main\Libs\reCAPTCHA;
	use Main\Services\SeoService;
	use Pecee\Http\Input\InputItem;
	use Pecee\Http\Request;
	use Pecee\Http\Response;
	use Pecee\Http\Url;
	use Pecee\SimpleRouter\SimpleRouter as Router;
	use Rakit\Validation\Validator;
	use Tamtamchik\SimpleFlash\Flash;
	
	/**
	 * Get url for a route by using either name/alias, class or method name.
	 *
	 * The name parameter supports the following values:
	 * - Route name
	 * - Controller/resource name (with or without method)
	 * - Controller class name
	 *
	 * When searching for controller/resource by name, you can use this syntax "route.name@method".
	 * You can also use the same syntax when searching for a specific controller-class "MyController@home".
	 * If no arguments is specified, it will return the url for the current loaded route.
	 *
	 * @param string|null $name
	 * @param string|array|null $parameters
	 * @param array|null $getParams
	 * @return \Pecee\Http\Url
	 * @throws \InvalidArgumentException
	 */
	function url(?string $name = null, $parameters = null, ?array $getParams = null): Url
	{
		if ($name === "home") {
			return new Url('/');
		}
		
		return Router::getUrl($name, $parameters, $getParams);
	}
	
	/**
	 * @return \Pecee\Http\Response
	 */
	function response(): Response
	{
		return Router::response();
	}
	
	/**
	 * @return \Pecee\Http\Request
	 */
	function request(): Request
	{
		return Router::request();
	}
	
	/**
	 * Get input class
	 * @param string|null $index Parameter index name
	 * @param string|mixed|null $defaultValue Default return value
	 * @param array ...$methods Default methods
	 * @return \Pecee\Http\Input\InputHandler|array|string|null
	 */
	function input($index = null, $defaultValue = null, ...$methods)
	{
		if ($index !== null) {
			return request()->getInputHandler()->value($index, $defaultValue, ...$methods);
		}
		
		return request()->getInputHandler();
	}

  /**
   * Set input value
   * @param string $key
   * @param mixed $value
   */
  function set_input(string $key, $value): void
  {
    request()->getInputHandler()->addPost($key, new InputItem($key, $value));
  }
	
	/**
	 * @param string $url
	 * @param int|null $code
	 */
	function redirect(string $url, ?int $code = null): void
	{
		if (rtrim($url, '/') == rtrim(url('auth.login'), '/')) {
			$url .= "?redirect=".$_SERVER["REQUEST_URI"];
		}
		
		// Clear old data
		if (isset($_SESSION['old'])) {
			unset($_SESSION['old']);
		}
		
		response()->redirect($url, $code);
	}
	
	/**
	 * Redirect back to previous page
	 * @return Redirect
	 */
	function back(): Redirect
	{
		return new Redirect(request()->getReferer());
	}
	
	/**
	 * Cookie handler function
	 */
	function cookies(): Cookie
	{
		static $instance = null;
		if (is_null($instance)) {
			$instance = new Cookie();
		}
		return $instance;
	}
	
	/**
	 * Session handler function
	 */
	function session(): Session
	{
		static $instance = null;
		if (is_null($instance)) {
			$instance = new Session();
		}
		return $instance;
	}
	
	/**
	 * Modules handler function
	 */
	function modules($name): ?Module
	{
		static $modules = null;
		static $instances = null;
		
		if (is_null($modules)) {
			$modules = db()->query("SELECT * FROM Modules");
			$modules = $modules->fetchAll(PDO::FETCH_ASSOC);

      $moduleSettings = db()->query("SELECT * FROM ModuleSettings");
      $moduleSettings = $moduleSettings->fetchAll(PDO::FETCH_ASSOC);
		}
		
		if (is_null($instances)) {
			foreach ($modules as $module) {
        // Get settings for the module
        $module['settings'] = [];
        foreach ($moduleSettings as $setting) {
            if ($setting['moduleID'] == $module['id']) {
                $module['settings'][$setting['name']] = $setting['value'];
            }
        }

				$instances[$module['slug']] = new Module(
					$module['id'],
					$module['slug'],
					$module['name'],
					"1.0.0",
					$module['isEnabled'],
					$module['settings']
				);
			}
		}
		return $instances[$name] ?? null;
	}
	
	/**
	 * Theme handler function
	 */
	function theme(): Theme
	{
		static $instance = null;
		if (is_null($instance)) {
			$startsWith = "theme:" . settings('themeName');
			$themeSettings = array_filter(settings(), function ($key) use ($startsWith) {
				return substr($key, 0, strlen($startsWith)) === $startsWith;
			}, ARRAY_FILTER_USE_KEY);
			
			$instance = new Theme(
				settings('themeName'),
				$themeSettings
			);
		}
		return $instance;
	}
	
	/*
	 * Template Engine
	 * */
	function template(): Engine
	{
		static $instance;
		if (is_null($instance)) {
			$instance = new Engine('../app/views/themes/' . settings('themeName'));
			$instance->loadExtension(new Asset('../app/views/themes/' . settings('themeName') . '/assets/', true));
		}
		
		return $instance;
	}
	
	function seo(): Seo
	{
		return SeoService::getSeo();
	}
	
	/**
	 * Validate data
	 * @param array $data
	 * @param bool $back
	 * @return array
	 */
	function validate(array $data = [], bool $back = true): array {
		$validator = new Validator;
		
		$validator->setMessages([
			'required' => t__('The :attribute is required'),
			'required_if' => t__('The :attribute is required'),
			'email' => t__('The :attribute is not valid email'),
			'numeric' => t__('The :attribute must be numeric'),
			'url' => t__('The :attribute is not valid url'),
			'same' => t__('The :attribute must be same with :field'),
			'uploaded_file' => t__('The :attribute is not valid uploaded file'),
			'date' => t__('The :attribute is not valid date format'),
			'ip' => t__('The :attribute is not valid IP Address'),
			'ipv4' => t__('The :attribute is not valid IPv4 Address'),
			'boolean' => t__('The :attribute must be a boolean'),
			'min' => t__('The :attribute minimum is :min'),
			'max' => t__('The :attribute maximum is :max'),
			'in' => t__('The :attribute only allows :allowed_values'),
			'not_in' => t__('The :attribute is not allowing :disallowed_values'),
			'between' => t__('The :attribute must be between :min and :max')
		]);
		
		$inputJSON = json_decode(file_get_contents('php://input'), true);
		$validation = $validator->validate($_POST + $_FILES + ($inputJSON ?? []), $data);
		if ($validation->fails()) {
			if ($back) {
				back()->flash("error", $validation->errors()->all());
			}
			return $validation->errors()->all();
		}
		return [];
	}
	
	/**
	 * Render view
	 * @param string $name
	 * @param array $data
	 * @return string
	 */
	function view(string $name, array $data = []): string
	{
		try {
			return template()->render('pages/'.str_replace('.', '/', $name), $data);
		} catch (Exception $e) {
			if (settings('debugModeStatus') == 1) {
				return $e->getMessage();
			} else {
				abort_404();
			}
		}
		
		return view('404');
	}
	
	function auth(): Auth
	{
		static $instance;
		if (is_null($instance)) {
			$instance = new Auth();
		}
		
		return $instance;
	}
	
	/**
	 * Get Database instance
	 * @return PDO
	 */
	function db(): PDO
	{
		global $db;
		return $db;
	}

  /**
   * Get markdown instance
   * @return GithubFlavoredMarkdownConverter
   */
  function markdown(): GithubFlavoredMarkdownConverter
  {
    static $instance;
    if (is_null($instance)) {
	    $instance = new GithubFlavoredMarkdownConverter([
		    'html_input' => 'strip',
		    'allow_unsafe_links' => false,
	    ]);
    }
		
		return $instance;
  }
	
	function forum(): Forum
	{
		static $instance = null;
		if (is_null($instance)) {
			$instance = new Forum();
		}
		
		return $instance;
	}
	
	/**
	 * Get settings
	 * @param string|null $key
	 * @param string|null $default
	 * @return string|array
	 */
	function settings(string $key = null, string $default = null)
	{
		static $settings = null;
		
		if (is_null($settings)) {
			$settingsData = db()->query("SELECT * FROM Settings");
			$settingsData = $settingsData->fetchAll();
			foreach ($settingsData as $setting) {
				$settings[$setting["name"]] = $setting["value"];
			}
		}
		
		if ($key === null) {
			return $settings;
		}
		
		if (isset($settings[$key])) {
			return $settings[$key];
		}
		
		return $default;
	}
	
	function lang()
	{
		global $lang;
		return $lang;
	}
	
	function recaptcha(): reCAPTCHA
	{
		static $instance;
		if (is_null($instance)) {
			$instance = new reCAPTCHA(modules('recaptcha')->settings('publicKey'), modules('recaptcha')->settings('privateKey'));
		}
		
		return $instance;
	}

/**
 * Abort request and render view with code
 * @param int $code
 * @param array $data
 */
	function abort(int $code, array $data = []) {
		response()->httpCode($code);
		echo view($code, $data);

		exit();
	}
	
	/**
	 * Abort request with 404 code
	 */
	function abort_404() {
		abort(404);
	}

/**
 * Abort request with 403 code
 * @param string|null $title
 * @param string|null $message
 */
  function abort_403(string $title = null, string $message = null) {
    abort(403, [
      'title' => $title,
      'message' => $message
    ]);
  }
	
	/**
	 * Abort request if bool equals true
	 * @param bool $bool
	 * @param int $code
	 */
	function abort_if(bool $bool, int $code) {
		if ($bool) abort($code);
	}
	
	/**
	 * Abort request if bool equals false
	 * @param bool $bool
	 * @param int $code
	 */
	function abort_unless(bool $bool, int $code) {
		if (!$bool) abort($code);
	}
	
	/**
	 * Abort request if perm not exist
	 * @param string|array $perm
	 */
	function abort_perm($perm) {
		abort_unless(auth()->user()->can($perm), 403);
	}
	
	/**
	 * Get flash message
	 * @param string|null $type
	 */
	function _message(?string $type = null): void
	{
		echo Flash::display($type);
	}
	
	/**
	 * Echo old input value
	 */
	function _old(string $input, $default = null, $escape = true)
	{
		echo $escape ? htmlspecialchars(old($input, $default)) : old($input, $default);
	}

  /**
   * Get old input value
   */
  function old(string $input, $default = null)
  {
    if (isset($_SESSION['old'][$input]))
      return $_SESSION['old'][$input];

    return $default;
  }
	
	/**
	 * Get current csrf-token
	 * @return string|null
	 */
	function csrf_token(): ?string
	{
		$baseVerifier = Router::router()->getCsrfVerifier();
		if ($baseVerifier !== null) {
			return strip_tags($baseVerifier->getTokenProvider()->getToken());
		}
		
		return null;
	}
	
	/**
	 * Print csrf input field
	 * @return void
	 */
	function _csrf(): void
	{
		echo '<input type="hidden" name="csrf_token" value="' . csrf_token() . '">';
	}
	
	/**
	 * Print method input field
	 * @param string $method
	 * @return void
	 */
	function _method(string $method = "GET" | "POST" | "PUT" | "PATCH" | "DELETE"): void
	{
		echo '<input type="hidden" name="_method" value="' . $method . '">';
	}
	
	/**
	 * Selected if value is true
	 * @param bool $bool
	 * @return void
	 */
	function _selected(bool $bool): void
	{
		if ($bool) echo 'selected';
	}
	
	/**
	 * Checked if value is true
	 * @param bool|null $bool
	 * @return void
	 */
	function _checked(?bool $bool): void
	{
		if ($bool) echo 'checked';
	}
	
	/**
	 * Disabled if value is true
	 * @param bool $bool
	 * @return void
	 */
	function _disabled(bool $bool): void
	{
		if ($bool) echo 'disabled';
	}
	
	/**
	 * Active if value is true
	 * @param bool $bool
	 * @return void
	 */
	function _active(bool $bool): void
	{
		if ($bool) echo 'active';
	}
	
	function slugify($value): string
	{
		static $slugify = null;
		if ($slugify == null) {
			$slugify = new Slugify([
				'rulesets' => ['default', 'turkish', 'russian', 'hindi', 'chinese'],
			]);
		}
		$slugify->addRule("+", " plus ");
		$slugify->addRule("&", " and ");
		$slug = $slugify->slugify($value);
		if (trim($slug) == "") {
			return uniqid();
		}
		return $slug;
	}
	
	function is_redirectable($url): bool
	{
		return preg_match("/^(\/)[a-zA-Z0-9_\-]+(\/|$)/", $url);
	}
	
	function post($parameter) {
		if (isset($_POST[$parameter])) {
			return htmlspecialchars(trim(strip_tags($_POST[$parameter])));
		}
		else {
			return false;
		}
	}
	
	function get($parameter) {
		if (isset($_GET[$parameter])) {
			return strip_tags(trim(addslashes($_GET[$parameter])));
		}
		else {
			return false;
		}
	}
	
	/**
	 * Limit string
	 *
	 * @param string $content
	 * @param int $limit
	 * @param string $more
	 * @return string
	 */
	function limit_string(string $content, int $limit = 0, string $more = '...'): string
	{
		$content = strip_tags($content);
		return mb_substr($content, 0, $limit, 'utf-8') . $more;
	}
	
	function datetime() {
		return date("Y-m-d H:i:s");
	}
	
	function escape_t__($string, $vars = [])
	{
		return str_replace("'", "\'", t__($string, $vars));
	}
	
	function escape_e__($string, $vars = [])
	{
		echo escape_t__($string, $vars);
	}
	
	function url_is($url): bool
	{
		return request()->getUrl() == url($url);
	}
	
	function validate_username($username): bool
	{
		$regex = settings('usernameRegex');
		return preg_match("/$regex/", $username) === 1;
	}
	
	function is_email($email): bool
	{
		return filter_var($email, FILTER_VALIDATE_EMAIL);
	}
	
	function validate_password($password): bool
	{
		if (strlen($password) < 4) return false;
		
		$badPasswordList = array(
			'1234',
			'12345',
			'123456',
			'1234567',
			'12345678',
			'123456789',
			'1234567890',
			'abc123',
			'xyz123',
			'qwerty',
			'qwerty123',
			'sifre',
			'sifre0',
			'sifre123',
			'password',
			'password0'
		);
		return !in_array($password, $badPasswordList);
	}
	
	function carbon($date = null): Carbon
	{
		static $factory = null;

		if (is_null($factory)) {
			$factory = new Factory([
				'locale' => lang() ?? settings('language'),
				'timezone' => settings("timezone"),
			]);
			
			Carbon::macro('intlFormat', function () {
				return $this->isoFormat('lll');
			});
		}
		
		// If date is timestamp
		if (is_numeric($date) && $date > 0) {
			// Check if it's in milliseconds
			if ($date > 9999999999) { // Larger than a 10-digit timestamp, assume its milliseconds
				$date = $date / 1000;
			}
			return $factory->make(Carbon::createFromTimestamp($date));
		}
		
		return $factory->make(($date === null) ? Carbon::now('UTC') : Carbon::parse($date));
	}
	
	/**
	 * @throws Exception
	 */
	function money(): Money
	{
		static $instance = null;
		global $currency;
		global $currencies;
		
		if (is_null($instance)) {
			$instance = new Money(settings('currency'), $currency["code"], $currencies);
		}
		
		return $instance;
	}
	
	function cache(): Cache
	{
		static $instance = null;
		
		if (is_null($instance)) {
			$instance = new Cache(settings());
		}
		
		return $instance;
	}
	
	function website_url($path = null): string
	{
		$baseURL = settings('websiteURL');
		if (settings('websiteURL') == '') {
			$baseURL = ((!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off' || $_SERVER['SERVER_PORT'] == 443) ? "https://" : "http://") . $_SERVER['HTTP_HOST'];
		}
		return rtrim($baseURL, '/') . ($path ? "/" . trim($path, '/') : "");
	}
	
	function shorten_number($number): string
	{
		if ($number < 1000) {
			return number_format($number);
		}
		else if ($number < 100000) {
			return number_format($number / 1000, 1)."K";
		}
		else if ($number < 1000000) {
			return number_format($number / 1000)."K";
		}
		else if ($number < 1000000000) {
			return number_format($number / 1000000)."M";
		}
		else {
			return number_format($number / 1000000000)."B";
		}
	}
	
	function escape($value): string
	{
		return htmlspecialchars($value, ENT_QUOTES, 'UTF-8');
	}
	
	function sanitize($value): string
	{
		return trim(strip_tags($value));
	}
	
	function rand_token(): string
	{
		return md5(uniqid(mt_rand(), true));
	}